<?php namespace DistrictHub\Api\Http\Controllers;

use DistrictHub\Directory\Church;
use DistrictHub\Directory\Member;
use App\Http\Controllers\Controller;
use App\Http\Requests;
use Geocoder\Geocoder;
use Illuminate\Http\Request;
use League\Fractal\Manager as Fractal;
use League\Fractal\Resource\Collection;

class ChurchController extends Controller
{
    /**
     * @var Fractal
     */
    private $fractal;

    /**
     * @param Fractal $fractal
     */
    public function __construct(Fractal $fractal)
    {
        $this->fractal = $fractal;
    }

    /**
     * Display a listing of the resource.
     *
     * @param Request  $request
     * @param Geocoder $geocoder
     *
     * @return Response
     */
    public function index(Request $request, Geocoder $geocoder)
    {
        // Get query string parameters
        $name = $request->input('name');
        $location = $request->input('location');
        $distance = $request->input('distance');
        $circuitId = $request->input('circuit_id');

        // Start database query
        $churches = Church::query();

        // Location based query
        if (!is_null($location)) {
            $geocodedLocation = $geocoder->geocode($location);

            if ($geocodedLocation->count() === 1) {
                $latitude = $geocodedLocation->first()->getLatitude();
                $longitude = $geocodedLocation->first()->getLongitude();

                $churches = $churches->select(\DB::raw("`directory_churches`.*,(3959*acos(cos(radians({$latitude}))*cos(radians(`directory_addresses`.`latitude`))*cos(radians(`directory_addresses`.`longitude`)-radians({$longitude}))+sin(radians({$latitude}))*sin(radians(`directory_addresses`.`latitude`)))) AS distance"))
                    ->join('directory_addresses', function ($join) {
                        $join->on('directory_churches.id', '=', 'directory_addresses.owner_id')
                            ->where('directory_addresses.owner_type', '=', Church::class);
                    })
                    ->orderBy('distance', 'asc');

                if ((int)$distance > 0) {
                    $churches = $churches->having('distance', '<', $distance);
                }
            }
        }

        // Name based query
        if (!is_null($name)) {
            $churches = $churches->where('name', 'LIKE', "%{$name}%");
        }

        // Circuit based query
        if (!is_null($circuitId)) {
            $churches = $churches->where('circuit_id', $circuitId);
        }

        // Execute query
        $churches = $churches->with('circuit')
            ->orderBy('name', 'asc')
            ->get();

        // Pass query results into collection to perform transform
        $resource = new Collection($churches, function (Church $church) {
            $address = "";
            if($church->address)
            {
                $address = implode(', ',array_filter(
                        [
                            $church->address->address_line1,
                            $church->address->address_line2,
                            $church->address->address_line3,
                            $church->address->city,
                            $church->address->postcode
                        ]
                    )
                );
            }

            return [
                'id'       => $church->id,
                'name'     => $church->name,
                'address'  => $address,
                'distance' => number_format($church->distance, 2),
                'picture'  => $church->picture->url('medium'),
                'link'     => $church->link,
                'circuit'  => [
                    'name' => $church->circuit->name
                ]
            ];
        });

        // Send JSON response with data
        return response()->json($this->fractal->createData($resource)->toArray());
    }

    /**
     * Display a listing of the resource.
     *
     * @param Request  $request
     * @param Geocoder $geocoder
     *
     * @return Response
     */
    public function ministers(Request $request, Geocoder $geocoder)
    {
        // Get query string parameters
        $name = $request->input('name');
        $church = $request->input('church');
        $location = $request->input('location');
        $distance = $request->input('distance');
        $circuitId = $request->input('circuit_id');
        $exact = $request->input('exact');

        // Start database query
        $members = Member::query();
        $selectColumns = [\DB::raw("`directory_members`.*")];

        // Church based query
        if (!is_null($church))
        {
            $members
                ->join('directory_member_role', function($join) {
                    $join->on('directory_members.id', '=', 'directory_member_role.member_id')
                        ->where('directory_member_role.scope_type', '=', Church::class);
                })
                ->join('directory_churches', function($join) use ($church){
                    $join->on('directory_member_role.scope_id', '=', 'directory_churches.id')
                        ->where('directory_churches.name', 'LIKE', "%{$church}%");
                });

            $selectColumns[] = \DB::raw('directory_churches.name as church');
        }


        // Location based query
        if (!is_null($location)) {
            $geocodedLocation = $geocoder->geocode($location);

            if ($geocodedLocation->count() === 1) {
                $latitude = $geocodedLocation->first()->getLatitude();
                $longitude = $geocodedLocation->first()->getLongitude();

                $members = $members->join('directory_addresses', function ($join) {
                        $join->on('directory_members.id', '=', 'directory_addresses.owner_id')
                            ->where('directory_addresses.owner_type', '=', Member::class)
                            ->where('directory_addresses.primary', '=', true);
                    })
                    ->orderBy('distance', 'asc');

                $selectColumns[] = \DB::raw("(3959*acos(cos(radians({$latitude}))*cos(radians(`directory_addresses`.`latitude`))*cos(radians(`directory_addresses`.`longitude`)-radians({$longitude}))+sin(radians({$latitude}))*sin(radians(`directory_addresses`.`latitude`)))) AS distance");

                if ((int)$distance > 0) {
                    $members = $members->having('distance', '<', $distance);
                }
            }
        }

        // Name based query
        if (!is_null($name)) {
            $nameParts = explode(' ', $name);

            $title = null;
            if(count($nameParts) == 3)
            {
                $title = isset($nameParts[0]) ? $nameParts[0] : null;
                $firstName = isset($nameParts[1]) ? $nameParts[1] : null;
                $lastName = isset($nameParts[2]) ? $nameParts[2] : $firstName;
            } else {
                $firstName = isset($nameParts[0]) ? $nameParts[0] : null;
                $lastName = isset($nameParts[1]) ? $nameParts[1] : $firstName;
            }


            if (!is_null($exact)) {
                $members = $members->where(function($query) use ($title,$firstName,$lastName){
                    $query->where('first_name', '=', $firstName)
                        ->where('last_name', '=', $lastName);
                    if(!is_null($title))
                    {
                        $query->where('title', '=', $title);
                    }
                });
            } else {
                $members = $members->where(function($query) use ($title,$firstName,$lastName){
                    $query->where('first_name', 'LIKE', "%{$firstName}%")
                        ->orWhere('last_name', 'LIKE', "%{$lastName}%");
                    if(!is_null($title))
                    {
                        $query->orWhere('title', 'LIKE', "%{$title}%");
                    }
                });
            }

        }

        if (!is_null($circuitId))
        {
            $members = $members->where('circuit_id', '=', $circuitId);
        }

        // Execute query
        $members = $members->where('type_id', '=', '1')
            ->orderBy('last_name', 'asc')
            ->orderBy('first_name', 'asc')
            ->get($selectColumns);

        // Pass query results into collection to perform transform
        $resource = new Collection($members, function (Member $member) use ($church) {
            $name = sprintf(
                '%s %s %s',
                $member->title,
                $member->first_name,
                $member->last_name
            );

            $minister = [
                'name'          => $name,
                'distance'      => number_format($member->distance, 2),
            ];

            $minister['church'] = null;
            if (!is_null($church))
            {
                $minister['church'] = $member->church;
            } else {
                if($member->church_roles) {
                    $churches = [];
                    foreach($member->church_roles as $role)
                    {
                        $churches[] = $role->pivot->scope->name;
                    }

                    $minister['church'] = join(', ', $churches);
                }
            }


            $minister['email_address'] = null;
            if($member->primary_email)
            {
                $minister['email_address'] = $member->primary_email->address;
            }

            $minister['phone_number'] = null;
            if($member->primary_phone)
            {
                $minister['phone_number'] = $member->primary_phone->number;
            }

            return $minister;
        });

        // Send JSON response with data
        return response()->json($this->fractal->createData($resource)->toArray());
    }
}
