<?php namespace DistrictHub\Auth\Http\Controllers;

use DistrictHub\Auth\Services\Activation;
use DistrictHub\Directory\Member;
use App\Http\Controllers\Controller;
use DistrictHub\Auth\Http\Requests\ActivationRequest;
use DistrictHub\Auth\Http\Requests\RegisterRequest;
use Carbon\Carbon;
use Illuminate\Contracts\Auth\Guard;
use Illuminate\Contracts\Routing\Registrar;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;

class AuthController extends Controller
{
    /*
    |--------------------------------------------------------------------------
    | Registration & Login Controller
    |--------------------------------------------------------------------------
    |
    | This controller handles the registration of new users, as well as the
    | authentication of existing users. By default, this controller uses
    | a simple trait to add these behaviors. Why don't you explore it?
    |
    */

    /**
     * The ID (or array of ID's) of the default role to assign to new Users.
     *
     * @var int|array
     */
    protected $defaultRole = 3;

    /**
     * The Guard implementation.
     *
     * @var Guard
     */
    protected $auth;

    /**
     * The registrar implementation.
     *
     * @var Registrar
     */
    protected $registrar;

    /**
     * Create a new authentication controller instance.
     *
     * @param  \Illuminate\Contracts\Auth\Guard     $auth
     * @param   $registrar
     *
     * @return void
     */
    public function __construct(Guard $auth, Registrar $registrar)
    {
        $this->auth = $auth;
        $this->registrar = $registrar;

        $this->middleware('guest', ['except' => 'getLogout']);
    }

    /**
     * Show the application registration form.
     *
     * @return \Illuminate\Http\Response
     */
    public function getRegister(Request $request)
    {
        $data = [];

        if($request->has('email')) {
            $email = $request->input('email');
            $members = Member::whereHas('emails', function(Builder $query) use ($email) {
                $query->where('address', $email);
            })->get();
            if($members->count() > 1) {
                $data['members'] = $members->pluck('name', 'id')->toArray();
            }
        }

        return view('auth::register', $data);
    }

    /**
     * Handle a registration request for the application.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function postRegister(RegisterRequest $request, Activation $activation)
    {
        $email = $request->input('email');
        $username = $request->input('username', $email);

        if($request->has('member')) {
            $member = Member::where('id', $request->input('member'));
        } else {
            $member = Member::whereHas('emails', function (Builder $query) use ($email) {
                $query->where('address', $email);
            });
        }

        if($member->count() > 1) {
            return redirect()->to('auth/register?email=' . $email);
        }

        $member = $member->first();

        $user = User::create([
            'name'     => $member->name,
            'username' => $username,
            'email'    => $member->primary_email->address,
            'active'   => false,
        ]);

        $user->roles()->attach($this->defaultRole);
        $user->member()->associate($member);
        $user->save();

        $activation->register($user);

        return redirect()->to('auth/register')->with('status', 'We have e-mailed your account activation link!');
    }

    public function getActivate($token = null)
    {
        if (is_null($token)) {
            throw new NotFoundHttpException;
        }

        return view('auth::activate', compact('token'));
    }

    public function postActivate(ActivationRequest $request)
    {
        $token = $request->input('token');
        $password = $request->input('password');

        $user = User::where('activate_token', $token)->first();

        if(is_null($user)) {
            return redirect()->back()->withErrors(['token' => 'Your activation token is invalid.']);
        }

        if($user->updated_at < Carbon::now()->subDays(3)) {
            return redirect()->back()->withErrors(['token' => 'Your activation token has expired.']);
        }

        $user->password = bcrypt($password);
        $user->active = true;
        $user->activate_token = null;

        $user->save();

        Auth::login($user);

        return redirect()->route('dashboard');
    }

    /**
     * Show the application login form.
     *
     * @return \Illuminate\Http\Response
     */
    public function getLogin()
    {
        return view('auth::login');
    }

    /**
     * Handle a login request to the application.
     *
     * @param  \Illuminate\Http\Request $request
     *
     * @return \Illuminate\Http\Response
     */
    public function postLogin(Request $request)
    {
        $this->validate($request, [
            'email'    => 'required',
            'password' => 'required',
        ]);

        $email = $request->input('email');
        $password = $request->input('password');

        $credentials = [
            'username' => $email,
            'password' => $password,
            'active'   => 1
        ];

        if ($this->auth->attempt($credentials, $request->has('remember'))) {
            return redirect()->intended($this->redirectPath());
        }

        $errors['email'] = 'These credentials do not match our records.';

        if ($this->auth->validate(['email' => $email, 'password' => $password])) {
            $errors['email'] = 'Your account has been disabled.';
        }

        return redirect('/auth/login')
            ->withInput($request->only('email'))
            ->withErrors($errors);
    }

    /**
     * Get the failed login message.
     *
     * @return string
     */
    protected function getFailedLoginMessage()
    {
        return 'These credentials do not match our records.';
    }

    /**
     * Log the user out of the application.
     *
     * @return \Illuminate\Http\Response
     */
    public function getLogout()
    {
        $this->auth->logout();

        return redirect('/');
    }

    /**
     * Get the post register / login redirect path.
     *
     * @return string
     */
    public function redirectPath()
    {
        if (property_exists($this, 'redirectPath'))
        {
            return $this->redirectPath;
        }

        return property_exists($this, 'redirectTo') ? $this->redirectTo : '/';
    }

    /**
     * Get the path to the login route.
     *
     * @return string
     */
    public function loginPath()
    {
        return property_exists($this, 'loginPath') ? $this->loginPath : '/auth/login';
    }
}
