<?php namespace DistrictHub\Auth;

use Carbon\Carbon;
use DistrictHub\Directory\Member;
use Illuminate\Auth\Authenticatable;
use Illuminate\Auth\Passwords\CanResetPassword;
use Illuminate\Contracts\Auth\Authenticatable as AuthenticatableContract;
use Illuminate\Contracts\Auth\CanResetPassword as CanResetPasswordContract;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Notifications\Notifiable;
use Zizaco\Entrust\Contracts\EntrustUserInterface;
use Zizaco\Entrust\Traits\EntrustUserTrait;

class User extends Model implements AuthenticatableContract, CanResetPasswordContract, EntrustUserInterface
{
    use Authenticatable, CanResetPassword, EntrustUserTrait, Notifiable;

    /**
     * The database table used by the model.
     *
     * @var string
     */
    protected $table = 'users';

    /**
     * The attributes that are mass assignable.
     *
     * @var array
     */
    protected $fillable = ['name', 'email', 'username', 'password', 'active'];

    /**
     * The attributes excluded from the model's JSON form.
     *
     * @var array
     */
    protected $hidden = ['password', 'remember_token'];

    /**
     * The related Member model.
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    public function member()
    {
        return $this->belongsTo(Member::class);
    }

    /**
     * Check group membership.
     *
     * @param $groupId
     *
     * @return bool
     */
    public function subscriberOf($groupId)
    {
        if (is_null($this->member_id)) {
            return false;
        }

        return $this->member->groups()->wherePivot('role_id', 3)->get()
            ->contains('id', $groupId);
    }

    /**
     * Check group membership.
     *
     * @param $groupId
     *
     * @return bool
     */
    public function memberOf($groupId)
    {
        if (is_null($this->member_id)) {
            return false;
        }

        return $this->member->groups()->wherePivot('role_id', '<>', 3)->get()
            ->contains('id', $groupId);
    }

    /**
     * Check group membership.
     *
     * @param $groupId
     *
     * @return bool
     */
    public function adminOf($groupId)
    {
        if (is_null($this->member_id)) {
            return false;
        }

        return ($this->member->groups()->wherePivot('role_id', 1)->get()->contains('id', $groupId)
            && $this->ability('district-officer', ['edit-own-group']))
            || $this->ability('district-admin', ['edit-group']);
    }

    /**
     * Attribute that returns the expiration date of the activation link or null.
     *
     * @return Carbon|null
     */
    public function getActivationLinkExpiresAtAttribute()
    {
        return $this->active ? null : $this->updated_at->addDay(3);
    }

    /**
     * Attribute that returns true if the expiration date has passed.
     *
     * @return bool
     */
    public function getActivationLinkExpiredAttribute()
    {
        return $this->active || $this->updated_at < Carbon::now()->subDays(3);
    }
}
