<?php namespace DistrictHub\Auth\Services;

use DistrictHub\Auth\Exceptions\UserAlreadyActiveException;
use DistrictHub\Auth\User;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use Illuminate\Mail\Message;
use Illuminate\Support\Facades\Mail;

/**
 * Service to handle user activation.
 *
 * @author       Tom Densham <tom.densham@studiobonito.co.uk>
 * @copyright    Studio Bonito Ltd.
 */
class Activation
{
    /**
     * Send the activation email for the user.
     *
     * @param User $user
     */
    public function register($user)
    {
        // Generate a new token
        $token = $this->generateActivationToken();

        // Set the token for the user
        $user->activate_token = $token;

        // Save the user
        $user->save();

        $this->sendActivationEmailTo($user, $token);
    }

    /**
     * Re-send the activation email with a new token if the user exists.
     *
     * @param User|int $user
     */
    public function resend($user)
    {
        if (is_numeric($user)) {
            // Find the user
            $user = User::find($user);
        }

        // Abort if no user is found
        if (!($user instanceof User)) {
            throw new ModelNotFoundException('Unable to find a user account that matches the request.');
        }

        if ($user->active) {
            throw (new UserAlreadyActiveException())->setUser($user);
        }

        // Re-send the activation link
        $this->register($user);
    }

    /**
     * Get all the user accounts who have failed to activate their account.
     *
     * @return \Illuminate\Support\Collection
     */
    public function usersWithExpiredTokens()
    {
        return User::where('active', false)->orderBy('updated_at')->get();
    }

    /**
     * Generate a new token.
     *
     * @return string
     */
    private function generateActivationToken()
    {
        return hash_hmac('sha256', str_random(40), config('app.key'));
    }

    /**
     * Send the activation link.
     *
     * @param $user
     * @param $token
     */
    private function sendActivationEmailTo($user, $token)
    {
        Mail::send(
            'auth::emails.activate',
            compact('token', 'user'),
            function (Message $message) use ($user, $token) {
                $message->subject('Your Account Activation Link');
                $message->to($user->email);
            }
        );
    }
}
