<?php namespace DistrictHub\Directory;

use DistrictHub\Groups\Group;
use DistrictHub\Auth\User;
use Illuminate\Database\Eloquent\Model;

class Member extends Model
{
    /**
     * Format the updated_at property.
     */
    use TimestampFormat;

    /**
     * The database table used by the model.
     *
     * @var string
     */
    protected $table = 'directory_members';

    /**
     * The attributes that are mass assignable.
     *
     * @var array
     */
    protected $fillable = [
        'title',
        'first_name',
        'last_name',
        'description',
        'comments',
        'circuit_id',
        'type_id',
    ];

    /**
     * The attributes excluded from the model's JSON form.
     *
     * @var array
     */
    protected $hidden = [
        'user_id',
    ];

    /**
     * The accessors to append to the model's array form.
     *
     * @var array
     */
    protected $appends = [
        'name',
    ];

    /**
     * The related Address models.
     *
     * @return \Illuminate\Database\Eloquent\Relations\MorphMany
     */
    public function addresses()
    {
        return $this->morphMany(Address::class, 'owner');
    }

    /**
     * The related Phone models.
     *
     * @return \Illuminate\Database\Eloquent\Relations\MorphMany
     */
    public function phones()
    {
        return $this->morphMany(Phone::class, 'owner');
    }

    /**
     * The related Email models.
     *
     * @return \Illuminate\Database\Eloquent\Relations\MorphMany
     */
    public function emails()
    {
        return $this->morphMany(Email::class, 'owner');
    }

    /**
     * The related Church models.
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function churches()
    {
        return $this->hasMany(Church::class, 'minister_id');
    }

    /**
     * The related Circuit model.
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    public function circuit()
    {
        return $this->belongsTo(Circuit::class);
    }

    /**
     * The related Group model.
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    public function group()
    {
        return $this->belongsTo(Group::class);
    }

    /**
     * The related MemberType model.
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    public function type()
    {
        return $this->belongsTo(MemberType::class);
    }

    /**
     * The related User model.
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasOne
     */
    public function user()
    {
        return $this->hasOne(User::class);
    }

    /**
     * The related Groups models.
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsToMany
     */
    public function groups()
    {
        return $this->belongsToMany(Group::class, 'group_members')->withPivot('role_id')->withTimestamps();
    }

    /**
     * The name attribute.
     *
     * @return string
     */
    public function getNameAttribute()
    {
        return sprintf('%s %s %s', $this->title, $this->first_name, $this->last_name);
    }

    /**
     * The primary_address attribute.
     *
     * @return Address|null
     */
    public function getPrimaryAddressAttribute()
    {
        if (!is_null($address = $this->addresses()->primary()->get()->first())) {
            return $address;
        }

        return null;
    }

    /**
     * The primary_phone attribute.
     *
     * @return Phone|null
     */
    public function getPrimaryPhoneAttribute()
    {
        if (!is_null($phone = $this->phones()->primary()->get()->first())) {
            return $phone;
        }

        return null;
    }

    /**
     * The primary_email attribute.
     *
     * @return Email|null
     */
    public function getPrimaryEmailAttribute()
    {
        if (!is_null($email = $this->emails()->primary()->get()->first())) {
            return $email;
        }

        return null;
    }

    /**
     * Directory Role related methods
     */

    /**
     * @return \Illuminate\Database\Eloquent\Relations\BelongsToMany
     */
    public function roles()
    {
        return $this->belongsToMany(Role::class, 'directory_member_role')
                    ->withPivot('id', 'scope_id', 'scope_type')->using(MemberRolePivot::class)
                    ->withTimestamps();
    }

    /**
     * @return \Illuminate\Database\Eloquent\Collection
     */
    public function getDistrictRolesAttribute()
    {
        return $this->roles()->wherePivot('scope_type', District::class)->get();
    }

    /**
     * @return \Illuminate\Database\Eloquent\Collection
     */
    public function getCircuitRolesAttribute()
    {
        return $this->roles()->wherePivot('scope_type', Circuit::class)->get();
    }

    /**
     * @return \Illuminate\Database\Eloquent\Collection
     */
    public function getChurchRolesAttribute()
    {
        return $this->roles()->wherePivot('scope_type', Church::class)->get();
    }

    /**
     * @param Member|array $member
     * @param string       $message
     */
    public function sendPrivateMessage($member, $message)
    {
        $messenger = app()->make(Services\Messenger::class);

        if (is_array($member) || $member instanceof \IteratorAggregate) {
            foreach ($member as $m) {
                $this->sendPrivateMessage($m, $message);
            }
        }

        if (is_a($member, Member::class) && $member !== $this && !empty($member->primary_email->address)) {
            $messenger->message($this->name, $member->primary_email->address, $message);
        }
    }
}
