<?php namespace DistrictHub\Directory;

use Carbon\Carbon;
use Illuminate\Database\Eloquent\Model;
use League\Flysystem\FileNotFoundException;
use Symfony\Component\HttpFoundation\File\UploadedFile;

class Church extends Model
{
    /**
     * Format the updated_at property.
     */
    use TimestampFormat;

    /**
     * The database table used by the model.
     *
     * @var string
     */
    protected $table = 'directory_churches';

    /**
     * The attributes that are mass assignable.
     *
     * @var array
     */
    protected $fillable = [
        'name',
        'picture',
        'link',
        'circuit_id',
        'ccli_number',
        'membership_count',
    ];

    /**
     * The attributes excluded from the model's JSON form.
     *
     * @var array
     */
    protected $hidden = [
        'directory_circuit_id',
        'membership_contact_id',
        'minister_id',
    ];

    /**
     * The related Address model.
     *
     * @return \Illuminate\Database\Eloquent\Relations\MorphOne
     */
    public function address()
    {
        return $this->morphOne(Address::class, 'owner');
    }

    /**
     * The related Phone model.
     *
     * @return \Illuminate\Database\Eloquent\Relations\MorphOne
     */
    public function phone()
    {
        return $this->morphOne(Phone::class, 'owner');
    }

    /**
     * The related Circuit model.
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    public function circuit()
    {
        return $this->belongsTo(Circuit::class);
    }

    /**
     * The related Role models
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsToMany
     */
    public function roles()
    {
        return $this->morphToMany(Role::class, 'scope', 'directory_member_role')
                    ->withPivot('member_id')->using(RoleMorphPivot::class)
                    ->withTimestamps()
                    ->orderBy('sort');
    }

    /**
     * Get picture attribute.
     *
     * @return Attachment
     */
    public function getPictureAttribute()
    {
        return new Attachment(
            'picture',
            $this->attributes['picture_file_name'],
            $this->attributes['picture_file_size'],
            $this->attributes['picture_content_type'],
            $this->attributes['picture_updated_at']
        );
    }

    /**
     * Set picture attribute
     *
     * @param mixed $value
     */
    public function setPictureAttribute($value)
    {
        if ($value === '') {
            list($type, $subtype) = explode('/', $this->attributes['picture_content_type']);

            try {
                \Storage::disk('s3')->delete([
                    $type . 's/picture/original/' . $this->attributes['picture_file_name'],
                    $type . 's/picture/medium/' . $this->attributes['picture_file_name'],
                    $type . 's/picture/thumb/' . $this->attributes['picture_file_name'],
                ]);
            } catch (FileNotFoundException $exception) {
            }

            $this->attributes['picture_file_name'] = null;
            $this->attributes['picture_file_size'] = null;
            $this->attributes['picture_content_type'] = null;
            $this->attributes['picture_updated_at'] = Carbon::now();
        }

        if ($value instanceof UploadedFile) {
            $clientMimeType = $value->getClientMimeType();
            list($type, $subtype) = explode('/', $clientMimeType);

            $this->attributes['picture_file_name'] = $value->getClientOriginalName();
            $this->attributes['picture_file_size'] = $value->getClientSize();
            $this->attributes['picture_content_type'] = $clientMimeType;
            $this->attributes['picture_updated_at'] = Carbon::now();

            \Storage::disk('s3')->put(
                '/' . $type . 's/picture/original/' . $this->attributes['picture_file_name'],
                file_get_contents($value->getRealPath()),
                'public'
            );

            \Storage::disk('s3')->put(
                '/' . $type . 's/picture/medium/' . $this->attributes['picture_file_name'],
                (string)\Image::make($value->getRealPath())->fit(300, 300)->encode('jpg', 100),
                'public'
            );

            \Storage::disk('s3')->put(
                '/' . $type . 's/picture/thumb/' . $this->attributes['picture_file_name'],
                (string)\Image::make($value->getRealPath())->fit(100, 100)->encode('jpg', 100),
                'public'
            );
        }
    }
}
