<?php namespace DistrictHub\Directory\Http\Controllers;

use DistrictHub\Directory\Jobs\CreateChurch;
use DistrictHub\Directory\Jobs\UpdateChurch;
use DistrictHub\Contracts\Gateways\Directory\ChurchGateway;
use DistrictHub\Directory\Church;
use DistrictHub\Directory\Circuit;
use App\Http\Controllers\Controller;
use DistrictHub\Directory\Http\Requests;
use DistrictHub\Directory\PhoneType;
use Illuminate\Http\Request;
use Laracasts\Flash\Flash;
use League\Csv\Reader;

class ChurchController extends Controller
{
    use ExportsCsv;

    /*
    |--------------------------------------------------------------------------
    | Directory Church Controller
    |--------------------------------------------------------------------------
    */

    /**
     * @var ChurchGateway
     */
    private $churches;

    /**
     * Create a new controller instance.
     *
     * @param ChurchGateway $churches
     */
    public function __construct(ChurchGateway $churches)
    {
        $this->churches = $churches;

        $this->middleware('auth');
    }

    /**
     * Display a listing of the resource.
     *
     * @return Response
     */
    public function index(Request $request)
    {
        $draw = $request->input('draw', 0);
        $columns = $request->input('columns', []);
        $order = $request->input('order', []);
        $start = $request->input('start', 0);
        $length = $request->input('length', 10);
        $fields = $request->only('name', 'circuit_id');

        array_walk($order, function (&$value, $key, $columns) {
            $value['column'] = $columns[$value['column']]['data'];
        }, $columns);

        $churches = $this->churches->search($fields, $order ?: [['column' => 'name', 'dir' => 'asc']]);

        // If AJAX request from DataList
        if ($request->wantsJson()) {
            $recordsTotal = $this->churches->total();
            $recordsFiltered = $churches->count();

            $churches = $churches->slice($start, $length);

            return response()->json(
                [
                    'draw'            => $draw,
                    'recordsTotal'    => $recordsTotal,
                    'recordsFiltered' => $recordsFiltered,
                    'data'            => $churches->values(),
                ]
            );
        } else {
            $churches = $churches->slice($start, $length);
        }

        return view('directory::churches.index', compact('churches', 'fields'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return Response
     */
    public function create()
    {
        return view('directory::churches.create');
    }

    /**
     * Store a newly created resource in storage.
     *
     * @return Response
     */
    public function store(Requests\ChurchRequest $request)
    {
        $this->dispatch(new CreateChurch($request->only(['name', 'link', 'picture', 'circuit_id'])));

        return redirect(route('churches.index'));
    }

    /**
     * Display the specified resource.
     *
     * @param  int $id
     *
     * @return Response
     */
    public function show($id)
    {
        $church = $this->churches->find($id);

        return view('directory::churches.show', compact('church'));
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int $id
     *
     * @return Response
     */
    public function edit($id)
    {
        $church = $this->churches->find($id);

        $types = PhoneType::all()->pluck('description', 'id');

        return view('directory::churches.edit', compact('church', 'types'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  int $id
     *
     * @return Response
     */
    public function update($id, Requests\ChurchRequest $request)
    {
        $this->dispatch(new UpdateChurch($id, $request->only(['name', 'link', 'picture', 'circuit_id', 'ccli_number', 'membership_count'])));

        return redirect()->back();
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int $id
     *
     * @return Response
     */
    public function destroy($id)
    {
        $member = Church::findOrFail($id);

        $member->delete();

        Flash::success("Deleted \"{$member->name}\"");

        return redirect()->back();
    }

    public function delete($churchId)
    {
        return view('directory::churches.delete', compact('churchId'));
    }

    public function import()
    {
        return view('directory::churches.import');
    }

    public function extract(Request $request)
    {
        $file = $request->file('csv');

        $reader = Reader::createFromPath($file->getPathName());

        $churches = $reader->fetchAssoc();

        foreach ($churches as $church) {
            if (isset($church['picture'])) {
                // TODO: Fix this and allow picture imports - may need to create asset objects
                unset($church['picture']);
            }
            if (isset($church['circuit'])) {
                $circuit = Circuit::where('name', $church['circuit'])->firstOrFail();
                $church['circuit_id'] = $circuit->id;
            }
            Church::create($church);
        }

        return redirect()->back();
    }

    /**
     * Export the searched resources as a CSV file.
     *
     * @param Request $request
     *
     * @return Response
     */
    public function export(Request $request)
    {
        $fields = $request->only('number', 'name');
        $order = $request->input('order', []);

        $columns = [
            'name',
            'link',
            'picture',
            'circuit',
            'address_line1',
            'address_line2',
            'address_line3',
            'city',
            'postcode',
            'created_at',
            'updated_at',
        ];

        $churches = $this->churches->search($fields, $order ?: [['column' => 'name', 'dir' => 'asc']])->toArray();

        $filename = 'churches.csv';

        $formatter = function ($row) use ($columns) {
            if (isset($row['picture_file_name'])) {
                $row['picture'] = $row['picture_file_name'];
            }

            if (isset($row['circuit_id'])) {
                $church = $this->churches->find($row['id']);
                $row['circuit'] = $church->circuit->name;
            }

            $output = [];

            foreach ($columns as $column) {
                if (isset($row[$column])) {
                    $output[$column] = $row[$column];
                } else {
                    $output[$column] = null;
                }
            }

            return $output;
        };

        return $this->generateCsv($churches, $columns, $filename, $formatter);
    }
}
