<?php namespace DistrictHub\Directory;

use Illuminate\Database\Eloquent\Model;

class Role extends Model
{
    /**
     * Format the updated_at property.
     */
    use TimestampFormat;

    /**
     * The database table used by the model.
     *
     * @var string
     */
    protected $table = 'directory_roles';

    /**
     * The attributes that are mass assignable.
     *
     * @var array
     */
    protected $fillable = [
        'name',
        'scope',
    ];

    /**
     * The accessors to append to the model's array form.
     *
     * @var array
     */
    protected $appends = [
        'scope_name',
        'edit_link',
    ];

    /**
     * Scope a query to return District roles.
     *
     * @param $query
     *
     * @return mixed
     */
    public function scopeDistrict($query)
    {
        return $query->where('scope', District::class);
    }

    /**
     * Scope a query to return Circuit roles.
     *
     * @param $query
     *
     * @return mixed
     */
    public function scopeCircuit($query)
    {
        return $query->where('scope', Circuit::class);
    }

    /**
     * Scope a query to return Church roles.
     *
     * @param $query
     *
     * @return mixed
     */
    public function scopeChurch($query)
    {
        return $query->where('scope', Church::class);
    }

    public function members()
    {
        return $this->belongsToMany(Member::class, 'directory_member_role')->withPivot('description');
    }

    public function groups()
    {
        return $this->belongsToMany(RoleGroup::class, 'directory_role_role_group');
    }

    /**
     * @return string
     */
    public function getScopeNameAttribute()
    {
        switch ($this->scope) {
            case District::class:
                return 'District';
                break;
            case Circuit::class:
                return 'Circuit';
                break;
            case Church::class:
                return 'Church';
                break;
            default:
                return 'Unknown';
        }
    }

    /**
     * @return string
     */
    public function getEditLinkAttribute()
    {
        return route('roles.edit', $this->id);
    }

    /**
     * Register model event bindings.
     */
    protected static function boot()
    {
        parent::boot();

        static::creating(function ($model) {
            $model->sort = static::where('scope', $model->scope)->max('sort') + 1;
        });
    }
}
