<?php namespace DistrictHub\Directory\Http\Controllers;

use DistrictHub\Directory\Http\Requests\RoleRequest;
use DistrictHub\Directory\Jobs\CreateRole;
use DistrictHub\Directory\Jobs\UpdateRole;
use DistrictHub\Contracts\Gateways\Directory\RoleGateway;
use App\Http\Controllers\Controller;
use Illuminate\Http\Request;

class RoleController extends Controller
{
    /*
    |--------------------------------------------------------------------------
    | Directory Role Controller
    |--------------------------------------------------------------------------
    */

    /**
     * @var RoleGateway
     */
    protected $roles;

    /**
     * Create a new controller instance.
     *
     * @param RoleGateway $roles
     */
    public function __construct(RoleGateway $roles)
    {
        $this->roles = $roles;

        $this->middleware('auth');
    }

    /**
     * Display a listing of the resource.
     *
     * @param Requests\Request $request
     *
     * @return Response
     */
    public function index(Request $request)
    {
        $draw = $request->input('draw', 0);
        $columns = $request->input('columns', []);
        $order = $request->input('order', []);
        $start = $request->input('start', 0);
        $length = $request->input('length', 10);
        $fields = $request->only('name', 'scope');

        array_walk($order, function (&$value, $key, $columns) {
            $value['column'] = $columns[$value['column']]['data'];
        }, $columns);

        $roles = $this->roles->search($fields, $order ?: [['column' => 'name', 'dir' => 'asc']]);

        // If AJAX request from DataList
        if ($request->wantsJson()) {
            $recordsTotal = $this->roles->total();
            $recordsFiltered = $roles->count();

            $roles = $roles->slice($start, $length);

            return response()->json(
                [
                    'draw'            => $draw,
                    'recordsTotal'    => $recordsTotal,
                    'recordsFiltered' => $recordsFiltered,
                    'data'            => $roles->values(),
                ]
            );
        } else {
            $roles = $roles->slice($start, $length);
        }

        return view('directory::roles.index', compact('roles', 'fields'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return Response
     */
    public function create()
    {
        return view('directory::roles.create');
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param \DistrictHub\Directory\Http\Requests\RoleRequest $request
     *
     * @return \DistrictHub\Directory\Http\Controllers\Response
     */
    public function store(RoleRequest $request)
    {
        $this->dispatch(new CreateRole($request->only([
            'name',
            'scope',
        ])));

        return redirect(route('roles.index'));
    }

    /**
     * Display the specified resource.
     *
     * @param  int $id
     *
     * @return Response
     */
    public function show($id)
    {
        $role = $this->roles->find($id);

        return view('directory::roles.show', compact('role'));
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int $id
     *
     * @return Response
     */
    public function edit($id)
    {
        $role = $this->roles->find($id);

        return view('directory::roles.edit', compact('role'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  int                                             $id
     * @param \DistrictHub\Directory\Http\Requests\RoleRequest $request
     *
     * @return \DistrictHub\Directory\Http\Controllers\Response
     */
    public function update($id, RoleRequest $request)
    {
        $this->dispatch(new UpdateRole($id, $request->all()));

        return redirect()->back();
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int $id
     *
     * @return Response
     */
    public function destroy($id)
    {
        //
    }
}
