<?php namespace DistrictHub\Directory\Http\Controllers;

use DistrictHub\Directory\Jobs\CreateCircuit;
use DistrictHub\Directory\Jobs\UpdateCircuit;
use DistrictHub\Contracts\Gateways\Directory\CircuitGateway;
use DistrictHub\Directory\Circuit;
use DistrictHub\Directory\District;
use App\Http\Controllers\Controller;
use DistrictHub\Directory\Http\Requests;
use Illuminate\Http\Request;
use League\Csv\Reader;
use League\Csv\Writer;
use SplTempFileObject;

class CircuitController extends Controller
{
    use ExportsCsv;

    /*
    |--------------------------------------------------------------------------
    | Directory Circuit Controller
    |--------------------------------------------------------------------------
    */

    /**
     * @var CircuitGateway
     */
    private $circuits;

    /**
     * Create a new controller instance.
     *
     * @param CircuitGateway $circuits
     */
    public function __construct(CircuitGateway $circuits)
    {
        $this->circuits = $circuits;

        $this->middleware('auth');
    }

    /**
     * Display a listing of the resource.
     *
     * @return Response
     */
    public function index(Request $request)
    {
        $draw = $request->input('draw', 0);
        $columns = $request->input('columns', []);
        $order = $request->input('order', []);
        $start = $request->input('start', 0);
        $length = $request->input('length', 10);
        $fields = $request->only('number', 'name');

        array_walk($order, function (&$value, $key, $columns) {
            $value['column'] = $columns[$value['column']]['data'];
        }, $columns);

        $circuits = $this->circuits->search($fields, $order ?: [['column' => 'name', 'dir' => 'asc']]);

        // If AJAX request from DataList
        if ($request->wantsJson()) {
            $recordsTotal = $this->circuits->total();
            $recordsFiltered = $circuits->count();

            $circuits = $circuits->slice($start, $length);

            return response()->json(
                [
                    'draw'            => $draw,
                    'recordsTotal'    => $recordsTotal,
                    'recordsFiltered' => $recordsFiltered,
                    'data'            => $circuits->values(),
                ]
            );
        }

        return view('directory::circuits.index', compact('circuits', 'fields'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return Response
     */
    public function create()
    {
        return view('directory::circuits.create');
    }

    /**
     * Store a newly created resource in storage.
     *
     * @return Response
     */
    public function store(Requests\CircuitRequest $request)
    {
        $this->dispatch(new CreateCircuit($request->only(['name', 'number'])));

        return redirect(route('circuits.index'));
    }

    /**
     * Display the specified resource.
     *
     * @param  int $id
     *
     * @return Response
     */
    public function show($id)
    {
        $circuit = $this->circuits->find($id);

        return view('directory::circuits.show', compact('circuit'));
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int $id
     *
     * @return Response
     */
    public function edit($id)
    {
        $circuit = $this->circuits->find($id);

        return view('directory::circuits.edit', compact('circuit'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  int $id
     *
     * @return Response
     */
    public function update($id, Requests\CircuitRequest $request)
    {
        $this->dispatch(new UpdateCircuit($id, $request->only(['name', 'number', 'charity_number', 'website'])));

        return redirect()->back();
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int $id
     *
     * @return Response
     */
    public function destroy($id)
    {
        //
    }

    public function import()
    {
        return view('directory::circuits.import');
    }

    public function extract(Request $request)
    {
        $file = $request->file('csv');

        $reader = Reader::createFromPath($file->getPathName());

        $circuits = $reader->fetchAssoc();

        foreach ($circuits as $circuit) {
            if (isset($circuit['district'])) {
                $district = District::where('name', $circuit['district'])->firstOrFail();
                $circuit['district_id'] = $district->id;
            }
            Circuit::create($circuit);
        }

        return redirect()->back();
    }

    /**
     * Export the searched resources as a CSV file.
     *
     * @param Request $request
     *
     * @return Response
     */
    public function export(Request $request)
    {
        $fields = $request->only('number', 'name');
        $order = $request->input('order', []);

        $columns = ['name', 'number', 'district', 'created_at', 'updated_at'];

        $circuits = $this->circuits->search($fields, $order ?: [['column' => 'number', 'dir' => 'asc']])->toArray();

        $filename = 'circuits.csv';

        $formatter = function ($row) use ($columns) {
            if (isset($row['district_id'])) {
                $circuit = $this->circuits->find($row['id']);
                $row['district'] = $circuit->district->name;
            }

            $output = [];

            foreach ($columns as $column) {
                if (isset($row[$column])) {
                    $output[$column] = $row[$column];
                }
            }

            return $output;
        };

        return $this->generateCsv($circuits, $columns, $filename, $formatter);
    }
}
