<?php namespace DistrictHub\Directory\Http\Controllers;

use DistrictHub\Directory\MemberType;
use DistrictHub\Directory\Jobs\CreateMember;
use DistrictHub\Directory\Jobs\LinkChurchToMember;
use DistrictHub\Directory\Jobs\UnlinkChurchFromMember;
use DistrictHub\Directory\Jobs\UpdateMember;
use DistrictHub\Directory\Jobs\AssignMemberToRole;
use DistrictHub\Directory\Jobs\RemoveMemberFromRole;
use DistrictHub\Contracts\Gateways\Directory\MemberGateway;
use DistrictHub\Directory\Address;
use DistrictHub\Directory\Circuit;
use DistrictHub\Directory\Email;
use DistrictHub\Directory\Member;
use DistrictHub\Directory\Phone;
use DistrictHub\Directory\Role;
use DistrictHub\Groups\Group;
use App\Http\Controllers\Controller;
use DistrictHub\Directory\Http\Requests;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Laracasts\Flash\Flash;
use League\Csv\Reader;

class MemberController extends Controller
{
    use ExportsCsv;

    /*
    |--------------------------------------------------------------------------
    | Directory Member Controller
    |--------------------------------------------------------------------------
    */

    /**
     * @var MemberGateway
     */
    protected $members;

    /**
     * Create a new controller instance.
     *
     * @return void
     */
    public function __construct(MemberGateway $members)
    {
        $this->members = $members;

        $this->middleware('auth');
    }

    /**
     * Display a listing of the resource.
     *
     * @return Response
     */
    public function index(Request $request)
    {
        $draw = $request->input('draw', 0);
        $columns = $request->input('columns', []);
        $order = $request->input('order', []);
        $start = $request->input('start', 0);
        $length = $request->input('length', 10);
        $fields = $request->only('title', 'first_name', 'last_name', 'type_id', 'circuit_id', 'role_id', 'scope_type');

        array_walk($order, function (&$value, $key, $columns) {
            $value['column'] = $columns[$value['column']]['data'];
        }, $columns);

        $members = $this->members->search(
            $fields,
            $order ?: [
                ['column' => 'last_name', 'dir' => 'asc'],
                ['column' => 'first_name', 'dir' => 'asc'],
            ]
        );

        // If AJAX request from DataList
        if ($request->wantsJson()) {
            $recordsTotal = $this->members->total();
            $recordsFiltered = $members->count();

            $members = $members->slice($start, $length);

            return response()->json(
                [
                    'draw'            => $draw,
                    'recordsTotal'    => $recordsTotal,
                    'recordsFiltered' => $recordsFiltered,
                    'data'            => $members->values(),
                ]
            );
        } else {
            $members = $members->slice($start, $length);
        }

        return view('directory::members.index', compact('members', 'fields'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return Response
     */
    public function create()
    {
        return view('directory::members.create');
    }

    /**
     * Store a newly created resource in storage.
     *
     * @return Response
     */
    public function store(Requests\MemberRequest $request)
    {
        $this->dispatch(new CreateMember($request->only([
            'title',
            'first_name',
            'last_name',
            'circuit_id',
            'type_id',
        ])));

        return redirect(route('members.index'));
    }

    /**
     * Display the specified resource.
     *
     * @param  int $id
     *
     * @return Response
     */
    public function show($id)
    {
        $member = $this->members->find($id);

        return view('directory::members.show', compact('member'));
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int $id
     *
     * @return Response
     */
    public function edit($id)
    {
        $member = $this->members->find($id);

        return view('directory::members.edit', compact('member'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  int $id
     *
     * @return Response
     */
    public function update($id, Requests\MemberRequest $request)
    {
        $this->dispatch(new UpdateMember($id, $request->all()));

        return redirect()->back();
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int $id
     *
     * @return Response
     */
    public function destroy($id)
    {
        $member = Member::findOrFail($id);

        $member->delete();

        Flash::success("Deleted \"{$member->name}\"");

        return redirect()->back();
    }

    public function delete($memberId)
    {
        return view('directory::members.delete', compact('memberId'));
    }

    public function verify($id)
    {
        $member = Member::findOrFail($id);

        $member->verified = true;

        $member->save();

        Flash::success("Verified \"{$member->name}\"");

        return redirect()->back();
    }

    public function linkChurch($id, Requests\LinkChurchRequest $request)
    {
        $this->dispatch(new LinkChurchToMember($request->input('church_id'), $id));

        return redirect()->back();
    }

    public function unlinkChurch($memberId, $churchId)
    {
        $this->dispatch(new UnlinkChurchFromMember($churchId, $memberId));

        return redirect()->back();
    }

    public function assign($id, Requests\AssignToRoleRequest $request)
    {
        $this->dispatch(new AssignMemberToRole(
            $id,
            $request->input('role_id'),
            $request->input('scope_id'),
            $request->input('scope_type'),
            $request->input('description')
        ));

        return redirect()->back();
    }

    public function remove($memberId, $roleId, Requests\RemoveFromRoleRequest $request)
    {
        $this->dispatch(new RemoveMemberFromRole(
            $memberId,
            $roleId,
            $request->input('scope_id'),
            $request->input('scope_type')
        ));

        return redirect()->back();
    }

    public function import()
    {
        return view('directory::members.import');
    }

    public function extract(Request $request)
    {
        $file = $request->file('csv');

        $reader = Reader::createFromPath($file->getPathName());

        $members = $reader->fetchAssoc();

        DB::beginTransaction();

        try {
            foreach ($members as $member) {
                if (isset($member['circuit']) && !empty($member['circuit'])) {
                    $member['circuit'] = trim($member['circuit']);
                    $circuit = Circuit::where('name', $member['circuit'])->firstOrFail();
                    $member['circuit_id'] = $circuit->id;
                    unset($member['circuit']);
                }

                $groupIds = [];

                if (isset($member['groups']) && !empty($member['groups'])) {
                    $groups = explode(';', trim($member['groups'], " \t\n\r\0\x0B;"));
                    foreach ($groups as $groupName) {
                        $group = Group::where('name', trim($groupName))->firstOrFail();
                        $groupIds[$group->id] = ['role_id' => 2];
                    }
                    unset($member['groups']);
                }

                $roleIds = [];

                if (isset($member['roles']) && !empty($member['roles'])) {
                    $roles = explode(';', trim($member['roles'], " \t\n\r\0\x0B;"));
                    foreach ($roles as $roleAndScope) {
                        $roleAndScope = explode(':', trim($roleAndScope, " \t\n\r\0\x0B:"));
                        $role = Role::where('name', trim($roleAndScope[0]))->firstOrFail();
                        $scopeClass = $role->scope;
                        $scope = $scopeClass::where('name', trim($roleAndScope[1]))->firstOrFail();
                        $roleIds[$role->id] = ['scope_id' => $scope->id, 'scope_type' => $scopeClass];
                    }
                    unset($member['roles']);
                }

                $address = null;

                if ((isset($member['address_line1']) && !empty($member['address_line1'])) ||
                    (isset($member['address_line2']) && !empty($member['address_line2'])) ||
                    (isset($member['address_line3']) && !empty($member['address_line3'])) ||
                    (isset($member['city']) && !empty($member['city'])) ||
                    (isset($member['postcode']) && !empty($member['postcode']))
                ) {
                    $address = new Address([
                        'address_line1' => trim($member['address_line1'], " \t\n\r\0\x0B,"),
                        'address_line2' => trim($member['address_line2'], " \t\n\r\0\x0B,"),
                        'address_line3' => trim($member['address_line3'], " \t\n\r\0\x0B,"),
                        'city'          => trim($member['city'], " \t\n\r\0\x0B,"),
                        'postcode'      => trim($member['postcode'], " \t\n\r\0\x0B,"),
                        'primary'       => true,
                    ]);
                }

                $phone = null;

                if (isset($member['number']) && !empty($member['number'])) {
                    $phone = new Phone([
                        'number'  => trim($member['number']),
                        'primary' => true,
                    ]);
                }

                $email = null;

                if (isset($member['email']) && !empty($member['email'])) {
                    $email = new Email([
                        'address' => trim($member['email']),
                        'primary' => true,
                    ]);
                }

                if (isset($member['type']) && !empty($member['type'])) {
                    $memberType = MemberType::where('description', trim($member['type']))->firstOrFail();
                    $member['type_id'] = $memberType->id;
                    unset($member['type']);
                }

                foreach ($member as $attribute) {
                    if (is_string($attribute)) {
                        $attribute = trim($attribute);
                    }
                }

                $conflicts = Member::where([
                    ['first_name', '=', $member['first_name']],
                    ['last_name', '=', $member['last_name']],
                ])->get();

                if ($conflicts->count() > 0) {
                    $pieces = $conflicts->pluck('id')->toArray();
                    $duplicates = implode(', ', $pieces);
                    \Log::warning("Import member: Potential duplicates [{$duplicates}]");
                }

                $record = Member::create($member);
                $record->groups()->attach($groupIds);
                $record->roles()->attach($roleIds);
                if ($address) {
                    $record->addresses()->save($address);
                }
                if ($phone) {
                    $record->phones()->save($phone);
                }
                if ($email) {
                    $record->emails()->save($email);
                }
            }
        } catch (\Exception $exception) {
            DB::rollBack();
            throw new \RuntimeException('Failed importing member', 500, $exception);
        }

        DB::commit();

        return redirect()->back();
    }

    /**
     * Export the searched resources as a CSV file.
     *
     * @param Request $request
     *
     * @return Response
     */
    public function export(Request $request)
    {
        $fields = $request->only('title', 'first_name', 'last_name', 'type_id', 'circuit_id', 'role_id', 'scope_type');
        $order = $request->input('order', []);

        $columns = [
            'title',
            'first_name',
            'last_name',
            'circuit',
            'groups',
            'roles',
            'type',
            'address_line1',
            'address_line2',
            'address_line3',
            'city',
            'postcode',
            'number',
            'email',
            'created_at',
            'updated_at',
        ];

        $members = $this->members->search($fields, $order ?: [['column' => 'first_name', 'dir' => 'asc']])->toArray();

        $filename = 'members.csv';

        $formatter = function ($row) use ($columns) {
            if (isset($row['circuit'])) {
                $row['circuit'] = $row['circuit']['name'];
            }

            if (isset($row['type'])) {
                $row['type'] = $row['type']['description'];
            }

            $groups = Group::whereHas('members', function ($query) use ($row) {
                $query->where('directory_members.id', $row['id']);
            })->get();

            $row['groups'] = $groups->implode('name', ';');

            $roles = DB::table('directory_member_role')
                       ->join('directory_roles', 'directory_member_role.role_id', '=', 'directory_roles.id')
                       ->where('member_id', $row['id'])->get();

            $rolesAndScopes = [];

            foreach ($roles as $role) {
                $scopeClass = $role->scope_type;
                $scope = $scopeClass::findOrFail($role->scope_id);
                $rolesAndScopes[] = "{$role->name}:{$scope->name}";
            }

            $row['roles'] = implode(';', $rolesAndScopes);

            $address = Address::where('owner_type', Member::class)
                              ->where('owner_id', $row['id'])
                              ->orderBy('primary', 'desc')
                              ->first();

            if ($address) {
                $row['address_line1'] = $address->address_line1;
                $row['address_line2'] = $address->address_line2;
                $row['address_line3'] = $address->address_line3;
                $row['city'] = $address->city;
                $row['postcode'] = $address->postcode;
            }

            $phone = Phone::where('owner_type', Member::class)
                          ->where('owner_id', $row['id'])
                          ->orderBy('primary', 'desc')
                          ->first();

            if ($phone) {
                $row['number'] = $phone->number;
            }

            $email = Email::where('owner_type', Member::class)
                          ->where('owner_id', $row['id'])
                          ->orderBy('primary', 'desc')
                          ->first();

            if ($email) {
                $row['email'] = $email->address;
            }

            $output = [];

            foreach ($columns as $column) {
                if (isset($row[$column])) {
                    $output[$column] = $row[$column];
                } else {
                    $output[$column] = null;
                }
            }

            return $output;
        };

        return $this->generateCsv($members, $columns, $filename, $formatter);
    }
}
