<?php namespace DistrictHub\Directory\Providers;

use DistrictHub\Directory\Address;
use DistrictHub\Directory\Email;
use DistrictHub\Directory\Events\ChurchLinkedToMember;
use DistrictHub\Directory\Events\ChurchUnlinkedFromMember;
use DistrictHub\Directory\Events\ChurchWasCreated;
use DistrictHub\Directory\Events\ChurchWasUpdated;
use DistrictHub\Directory\Events\CircuitWasCreated;
use DistrictHub\Directory\Events\CircuitWasUpdated;
use DistrictHub\Directory\Events\MemberAssignedToRole;
use DistrictHub\Directory\Events\MemberRemovedFromRole;
use DistrictHub\Directory\Events\MemberWasCreated;
use DistrictHub\Directory\Events\MemberWasUpdated;
use DistrictHub\Directory\Events\RoleWasCreated;
use DistrictHub\Directory\Events\RoleWasUpdated;
use DistrictHub\Directory\Listeners\AlertChurchCreated;
use DistrictHub\Directory\Listeners\AlertChurchLinkedToMember;
use DistrictHub\Directory\Listeners\AlertChurchUnlinkedFromMember;
use DistrictHub\Directory\Listeners\AlertChurchUpdated;
use DistrictHub\Directory\Listeners\AlertCircuitCreated;
use DistrictHub\Directory\Listeners\AlertCircuitUpdated;
use DistrictHub\Directory\Listeners\AlertMemberAssignedToRole;
use DistrictHub\Directory\Listeners\AlertMemberCreated;
use DistrictHub\Directory\Listeners\AlertMemberRemovedFromRole;
use DistrictHub\Directory\Listeners\AlertMemberUpdated;
use DistrictHub\Directory\Listeners\AlertRoleCreated;
use DistrictHub\Directory\Listeners\AlertRoleUpdated;
use DistrictHub\Directory\Member;
use DistrictHub\Directory\Phone;
use DistrictHub\Directory\Role;
use DistrictHub\Directory\RoleObserver;
use Geocoder\Geocoder;
use Geocoder\Provider\GoogleMaps;
use Illuminate\Foundation\Support\Providers\EventServiceProvider as ServiceProvider;
use Illuminate\Support\Facades\DB;
use Ivory\HttpAdapter\CurlHttpAdapter;

class EventServiceProvider extends ServiceProvider
{
    /**
     * The event handler mappings for the application.
     *
     * @var array
     */
    protected $listen = [
        CircuitWasCreated::class        => [
            AlertCircuitCreated::class,
        ],
        CircuitWasUpdated::class        => [
            AlertCircuitUpdated::class,
        ],
        ChurchWasCreated::class         => [
            AlertChurchCreated::class,
        ],
        ChurchWasUpdated::class         => [
            AlertChurchUpdated::class,
        ],
        MemberWasCreated::class         => [
            AlertMemberCreated::class,
        ],
        MemberWasUpdated::class         => [
            AlertMemberUpdated::class,
        ],
        MemberAssignedToRole::class     => [
            AlertMemberAssignedToRole::class,
        ],
        MemberRemovedFromRole::class    => [
            AlertMemberRemovedFromRole::class,
        ],
        ChurchLinkedToMember::class     => [
            AlertChurchLinkedToMember::class,
        ],
        ChurchUnlinkedFromMember::class => [
            AlertChurchUnlinkedFromMember::class,
        ],
        RoleWasCreated::class           => [
            AlertRoleCreated::class,
        ],
        RoleWasUpdated::class           => [
            AlertRoleUpdated::class,
        ],
    ];

    public function register()
    {
        $this->app->singleton(Geocoder::class, function() {
            $httpAdapter = new CurlHttpAdapter();
            return new GoogleMaps($httpAdapter, null, null, true, config('services.google.key'));
        });
    }

    /**
     * Register any other events for your application.
     *
     * @return void
     */
    public function boot()
    {
        parent::boot();

        Role::observe(RoleObserver::class);

        // Enforce a single primary record per owner
        Address::saving(function (Address $address) {
            if ($address->exists && $address->primary == true) {
                DB::table($address->getTable())
                  ->where('primary', true)
                  ->where('id', '<>', $address->id)
                  ->where('owner_id', $address->owner_id)
                  ->where('owner_type', $address->owner_type)
                  ->update(['primary' => false]);
            }
        });

        // Enforce a single primary record per owner
        Email::saving(function (Email $email) {
            if ($email->exists && $email->primary == true) {
                DB::table($email->getTable())
                  ->where('primary', true)
                  ->where('id', '<>', $email->id)
                  ->where('owner_id', $email->owner_id)
                  ->where('owner_type', $email->owner_type)
                  ->update(['primary' => false]);
            }
        });

        // Enforce a single primary record per owner
        Phone::saving(function (Phone $phone) {
            if ($phone->exists && $phone->primary == true) {
                DB::table($phone->getTable())
                  ->where('primary', true)
                  ->where('id', '<>', $phone->id)
                  ->where('owner_id', $phone->owner_id)
                  ->where('owner_type', $phone->owner_type)
                  ->update(['primary' => false]);
            }
        });

        // Update User name when updating Member name
        Member::saving(function (Member $member) {
            if ($member->exists && $member->user) {
                $member->user->name = $member->name;
                $member->user->save();
            }
        });

        // Update User email when updating Member email
        Email::saving(function (Email $email) {
            if ($email->exists && $email->primary
                && $email->owner && $email->owner->user
            ) {
                if ($email->owner->user->username === $email->owner->user->email) {
                    $email->owner->user->username = $email->address;
                }
                $email->owner->user->email = $email->address;
                $email->owner->user->save();
            }
        });

        // Update Address coordinates when saving
        Address::saving(function (Address $address) {
            if ($address->exists) {
                $geocoder = $this->app['Geocoder\Geocoder'];
                $geocodedAddress = $geocoder->geocode($address->city . ' ' . $address->postcode)->first();
                $address->latitude = $geocodedAddress->getLatitude();
                $address->longitude = $geocodedAddress->getLongitude();
            }
        });
    }
}
