<?php namespace DistrictHub\Directory\Gateways;

use DistrictHub\Contracts\Gateways\Directory\CircuitGateway as CircuitGatewayContract;
use DistrictHub\Directory\Gateways\CachedGateway;
use Illuminate\Contracts\Cache\Repository as Cache;

/**
 * CachedCircuitGateway.
 *
 * @author       Tom Densham <tom.densham@studiobonito.co.uk>
 * @copyright    Studio Bonito Ltd.
 * @package      App\Gateways\Directory
 */
class CachedCircuitGateway extends CachedGateway implements CircuitGatewayContract
{
    /**
     * @var CircuitGatewayContract
     */
    private $gateway;

    /**
     * Create new gateway instance.
     *
     * @param CircuitGatewayContract $gateway
     * @param Cache                  $cache
     *
     */
    public function __construct(CircuitGatewayContract $gateway, Cache $cache)
    {
        parent::__construct($cache);
        $this->gateway = $gateway;
    }

    /**
     * Search records with fields and sort with order.
     *
     * @param array $fields
     * @param array $order
     *
     * @return \Illuminate\Database\Eloquent\Collection|static[]
     */
    public function search(array $fields, array $order)
    {
        return $this->remember(
            $this->cacheKey(__METHOD__, func_get_args()),
            function () use ($fields, $order) {
                return $this->gateway->search($fields, $order);
            }
        );
    }

    /**
     * Get the total number of records.
     *
     * @return int
     */
    public function total()
    {
        return $this->remember(
            $this->cacheKey(__METHOD__),
            function () {
                return $this->gateway->total();
            }
        );
    }

    /**
     * Find a model by its primary key.
     *
     * @param  mixed $id
     *
     * @return \DistrictHub\Directory\Circuit|null
     */
    public function find($id)
    {
        return $this->remember(
            $this->cacheKey(__METHOD__, func_get_args()),
            function () use ($id) {
                return $this->gateway->find($id);
            }
        );
    }

    /**
     * Update a model with the specified primary key.
     *
     * @param int   $id
     * @param array $attributes
     *
     * @return bool
     */
    public function update($id, array $attributes)
    {
        if ($this->gateway->update($id, $attributes)) {
            $this->cache()->flush();

            return true;
        }

        return false;
    }

    /**
     * Get array of ids/values for use in select fields.
     *
     * @param string $valueColumn
     * @param string $idColumn
     * @param string $default
     *
     * @return array
     */
    public function options($valueColumn, $idColumn = 'id', $default = '-- Select --')
    {
        return $this->remember(
            $this->cacheKey(__METHOD__, func_get_args()),
            function () use ($valueColumn, $idColumn, $default) {
                return $this->gateway->options($valueColumn, $idColumn, $default);
            }
        );
    }
}
