<?php namespace DistrictHub\Directory\Http\Controllers;

use App\Http\Controllers\Controller;
use DistrictHub\Directory\Church;
use DistrictHub\Directory\Email;
use DistrictHub\Directory\EmailType;
use DistrictHub\Directory\Http\Requests;
use DistrictHub\Directory\Member;
use Illuminate\Http\Request;
use Laracasts\Flash\Flash;

class EmailController extends Controller
{
    /*
    |--------------------------------------------------------------------------
    | Directory Email Controller
    |--------------------------------------------------------------------------
    */

    /**
     * Create a new controller instance.
     *
     * @return void
     */
    public function __construct()
    {
        $this->middleware('auth');
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return Response
     */
    public function create(Request $request)
    {
        $memberId = $request->route()->parameter('member');
        $churchId = $request->route()->parameter('church');

        if (!is_null($memberId)) {
            $owner = Member::find($memberId);
        }

        if (!is_null($churchId)) {
            $owner = Church::find($churchId);
        }

        $types = EmailType::all()->pluck('description', 'id');

        return view('directory::emails.create', compact('types', 'owner'));
    }

    /**
     * Store a newly created resource in storage.
     *
     * @return Response
     */
    public function store($id, Requests\EmailRequest $request)
    {
        $email = Email::create($request->all());

        $memberId = $request->route()->parameter('member');
        $churchId = $request->route()->parameter('church');

        if (!is_null($memberId)) {
            $member = Member::find($memberId);
            $member->emails()->save($email);

            $member->logActivity('updated', [
                'attributes' => ['email.address' => $email->address],
                'old'        => ['email.address' => ''],
            ]);

            if ($email->primary) {
                $member->primary_email_id = $email->id;
                $member->save();
            }
        }

        if (!is_null($churchId)) {
            $church = Church::find($churchId);
            $church->email()->save($email);
        }

        Flash::success("Created Email \"{$email->address}\"");

        return redirect()->back();
    }

    /**
     * Display the specified resource.
     *
     * @param  int $id
     *
     * @return Response
     */
    public function show(Request $request)
    {
        $emailId = $request->route()->parameter('email');

        $email = Email::find($emailId);

        return view('directory::emails.show', compact('email'));
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @return Response
     */
    public function edit(Request $request)
    {
        $emailId = $request->route()->parameter('email');

        $email = Email::find($emailId);

        $memberId = $request->route()->parameter('member');
        $churchId = $request->route()->parameter('church');

        if (!is_null($memberId)) {
            $owner = Member::find($memberId);
        }

        if (!is_null($churchId)) {
            $owner = Church::find($churchId);
        }

        $types = EmailType::all()->pluck('description', 'id');

        return view('directory::emails.edit', compact('email', 'types', 'owner'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @return Response
     */
    public function update(Member $member, Email $email, Requests\EmailRequest $request)
    {
        $email->fill($request->all());

        $original = $email->getOriginal();
        $dirty = $email->getDirty();

        $properties = [];

        if (key_exists('address', $dirty)) {
            $properties['attributes']['email.address'] = $dirty['address'];
            $properties['old']['email.address'] = $original['address'];
        }

        $email->save();

        $member->logActivity('updated', $properties);

        if ($email->primary && $member->primary_email_id != $email->id) {
            $member->primary_email_id = $email->id;
            $member->save();
        }

        Flash::success("Updated Email \"{$email->address}\"");

        return redirect()->back();
    }

    /**
     * Show delete confirmation.
     *
     * @return Response
     */
    public function delete($memberId, $emailId)
    {
        return view('directory::emails.delete', compact('memberId', 'emailId'));
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int $id
     *
     * @return Response
     */
    public function destroy(Member $member, $emailId)
    {
        $email = Email::find($emailId);

        Email::destroy($emailId);

        $member->logActivity('updated', [
            'attributes' => ['email.address' => ''],
            'old'        => ['email.address' => $email->address],
        ]);

        Flash::success("Deleted Email \"{$email->address}\"");

        return redirect()->back();
    }
}
