<?php

namespace DistrictHub\Directory;

use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Model;
use Spatie\EloquentSortable\Sortable;
use Spatie\EloquentSortable\SortableTrait;

class RoleGroup extends Model implements Sortable
{
    /**
     * Format the updated_at property.
     */
    use TimestampFormat;

    use SortableTrait;

    public $sortable = [
        'order_column_name'  => 'sort',
        'sort_when_creating' => true,
    ];

    /**
     * The database table used by the model.
     *
     * @var string
     */
    protected $table = 'directory_role_groups';

    /**
     * The attributes that are mass assignable.
     *
     * @var array
     */
    protected $fillable = [
        'name',
        'parent_id',
    ];

    /**
     * The accessors to append to the model's array form.
     *
     * @var array
     */
    protected $appends = [
        'create_link',
        'show_link',
        'edit_link',
        'destroy_link',
    ];

    /**
     * Scope role groups by circuit.
     *
     * @param \Illuminate\Database\Eloquent\Builder $query
     * @param \DistrictHub\Directory\Circuit        $circuit
     *
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function scopeCircuit(Builder $query, Circuit $circuit)
    {
        return $query->where('scope_type', '=', Circuit::class)->where('scope_id', '=', $circuit->id);
    }

    /**
     * @param \Illuminate\Database\Eloquent\Builder $query
     *
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function scopeTopLevel(Builder $query)
    {
        return $query->where('parent_id', '=', 0);
    }

    public function scope()
    {
        return $this->morphTo();
    }

    public function roleGroups()
    {
        return $this->hasMany(RoleGroup::class, 'parent_id')->ordered();
    }

    public function roles()
    {
        return $this->belongsToMany(Role::class, 'directory_role_role_group');
    }

    public function getCreateLinkAttribute()
    {
        return route('circuits.role-groups.create', $this->scope_id);
    }

    public function getShowLinkAttribute()
    {
        return route('circuits.role-groups.show', [$this->scope_id, $this->id]);
    }

    public function getEditLinkAttribute()
    {
        return route('circuits.role-groups.edit', [$this->scope_id, $this->id]);
    }

    public function getDestroyLinkAttribute()
    {
        return route('circuits.role-groups.destroy', [$this->scope_id, $this->id]);
    }

    /**
     * Build eloquent builder of sortable.
     *
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function buildSortQuery()
    {
        return static::query()->where('parent_id', '=', $this->parent_id);
    }
}
