@extends('app')

@section('title')
    <h2>
        Members
        @if(Auth::user()->ability('district-admin', 'create-member'))
            <a class="btn btn-default pull-right" href="{{ route('members.create') }}">
                <i class="fa fa-plus-circle"></i>
                Add Member
            </a>
        @endif
    </h2>
@stop

@section('breadcrumbs')
    <ol class="breadcrumb breadcrumb-plain flush-bottom">
        <li><a href="/"><i class="fa fa-home visible-xs"></i> <span class="hidden-xs">Dashboard</span></a></li>
        <li class="active">Members</li>
    </ol>
@stop

@section('content')
    <div class="panel panel-default">
        <div class="panel-heading">Search Members</div>
        <div class="panel-body">
            {!! Form::model($fields, ['method' => 'GET']) !!}
            @include('directory::members.fields.search', ['submitButtonText' => 'Search'])
            <a class="btn btn-default" href="{{ route('members.index') }}">Clear</a>
            {!! Form::close() !!}
        </div>
    </div>

    <div class="panel panel-default">
        <div class="panel-heading">
            Members
            @if(Auth::user()->ability('district-admin', 'export-member'))
                <div class="btn-group pull-right">
                    <button type="button" class="btn btn-default btn-xs dropdown-toggle" data-toggle="dropdown"
                            aria-expanded="false">
                        Actions <span class="caret"></span>
                    </button>
                    <ul class="dropdown-menu" role="menu">
                        <li>
                            <a href="{{ route('members.import') }}" data-toggle="modal" data-target="#modal">
                                <i class="fa fa-upload"></i>
                                Import
                            </a>
                        </li>
                        <li>
                            <a href="{{ route('members.export') }}?title={{ Request::input('title') }}&first_name={{ Request::input('first_name') }}&last_name={{ Request::input('last_name') }}&type_id={{ Request::input('type_id') }}&circuit_id={{ Request::input('circuit_id') }}&role_id={{ Request::input('role_id') }}&scope_type={{ Request::input('scope_type') }}&order={{ Request::input('order') }}">
                                <i class="fa fa-download"></i>
                                Export
                            </a>
                        </li>
                    </ul>
                </div>
            @endif
        </div>

        <table id="members-table" class="table table table-striped table-bordered table-hover display" cellspacing="0"
               width="100%">
            <thead>
                <tr>
                    <th>Title</th>
                    <th>First Name</th>
                    <th>Last Name</th>
                    <th>Type</th>
                    <th>Circuit</th>
                    <th>Last Updated</th>
                    <th></th>
                </tr>
            </thead>

            @foreach($members as $member)
                <tr>
                    <td>{{ $member->title }}</td>
                    <td>{{ $member->first_name }}</td>
                    <td>{{ $member->last_name }}</td>
                    <td>{{ $member->type->description }}</td>
                    <td>{{ $member->circuit->name }}</td>
                    <td>{{ $member->updated_at }}</td>
                    <td data-display="{{ $member->id }}">
                        <a class="btn btn-default btn-xs" href="{{ route('members.show', ['id' => $member->id]) }}"
                           data-toggle="tooltip" data-placement="top" title="View member">
                            <i class="fa fa-eye"></i>
                        </a>
                        @if(Auth::user()->ability('district-admin', 'edit-member'))
                            <a class="btn btn-default btn-xs" href="{{ route('members.edit', ['id' => $member->id]) }}"
                               data-toggle="tooltip" data-placement="top" title="Edit member">
                                <i class="fa fa-pencil"></i>
                            </a>
                            <a class="btn btn-default btn-xs"
                               href="{{ route('members.delete', ['id' => $member->id]) }}" data-toggle="modal"
                               data-target="#modal" title="Delete member">
                                <i class="fa fa-trash"></i>
                            </a>
                        @endif
                    </td>
                </tr>
            @endforeach
        </table>
    </div>
@endsection

@section('scripts')
    <script src="//cdn.datatables.net/1.10.4/js/jquery.dataTables.min.js"></script>
    <script src="//cdn.datatables.net/plug-ins/3cfcc339e89/integration/bootstrap/3/dataTables.bootstrap.js"></script>
    <script>
        $(document).ready(function () {
            var table = $('#members-table').DataTable({
                language: {
                    processing: '<img class="processing" width="64px" height="64px" src="data:image/gif;base64,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"/>',
                },
                dom: 'lfr<"table-responsive"t><"panel-footer"<"row"<"col-sm-5"i><"col-sm-7"p>>>',
                lengthChange: false,
                searching: false,
                processing: true,
                serverSide: true,
                ajax: '{!! Request::fullUrl() !!}',
                order: [[2, 'asc'], [1, 'asc']],
                columns: [
                    {
                        data: 'title',
                        className: 'data',
                        orderable: false,
                    },
                    {data: 'first_name', className: 'data'},
                    {data: 'last_name', className: 'data'},
                    {data: 'type.description', className: 'data'},
                    {data: 'circuit.name', className: 'data'},
                    {data: 'updated_at', className: 'data'},
                    {
                        data: 'id',
                        orderable: false,
                        defaultContent: '',
                        render: function (data) {
                            if ($(data).filter('a').length !== 2) {
                                @if(Auth::user()->ability('district-admin', 'edit-member'))
                                    return '<a class="btn btn-default btn-xs" href="/members/' + data + '" data-toggle="tooltip" data-placement="top" title="View member"><i class="fa fa-eye"></i></a>'
                                    + ' <a class="btn btn-default btn-xs" href="/members/' + data + '/edit" data-toggle="tooltip" data-placement="top" title="Edit member"><i class="fa fa-pencil"></i></a>'
                                    + ' <a class="btn btn-default btn-xs" href="/members/' + data + '/delete" data-toggle="modal" data-target="#modal" title="Delete member"><i class="fa fa-trash"></i></a>';
                                @else
                                    return '<a class="btn btn-default btn-xs" href="/members/' + data + '" data-toggle="tooltip" data-placement="top" title="View member"><i class="fa fa-eye"></i></a>';
                                @endif
                            } else {
                                return data;
                            }
                        },
                    },
                ],
            });

            table.on('draw.dt', function () {
                $('[data-toggle="tooltip"]').tooltip();
            });

            $('#members-table tbody').on('click', 'td.data', function () {
                var data = table.row($(this).parent()[0]).data();
                window.location = '/members/' + data.id;
            });
        });
    </script>
@stop