<?php namespace DistrictHub\Directory;

use Carbon\Carbon;
use DistrictHub\Groups\Group;
use DistrictHub\Auth\User;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Notifications\Notifiable;
use Spatie\Activitylog\ActivityLogger;
use Spatie\Activitylog\Traits\LogsActivity;

class Member extends Model
{
    /**
     * Log model updates.
     */
    use LogsActivity;

    /**
     * Enable notifications.
     */
    use Notifiable;

    /**
     * Format the updated_at property.
     */
    use TimestampFormat;

    /**
     * The database table used by the model.
     *
     * @var string
     */
    protected $table = 'directory_members';

    /**
     * The attributes that are mass assignable.
     *
     * @var array
     */
    protected $fillable = [
        'title',
        'first_name',
        'last_name',
        'description',
        'comments',
        'circuit_id',
        'type_id',
        'primary_address_id',
        'primary_email_id',
        'primary_phone_id',
        'receives_district_notifications',
        'receives_circuit_notifications',
    ];

    /**
     * The attributes excluded from the model's JSON form.
     *
     * @var array
     */
    protected $hidden = [
        'user_id',
    ];

    /**
     * The accessors to append to the model's array form.
     *
     * @var array
     */
    protected $appends = [
        'name',
    ];

    protected static $recordEvents = ['updated'];

    protected static $logOnlyDirty = true;

    protected static $logAttributes = [
        'title',
        'first_name',
        'last_name',
        'description',
        'comments',
        'circuit.name',
        'type.description',
        'primary_address.address_line1',
        'primary_address.address_line2',
        'primary_address.address_line3',
        'primary_address.city',
        'primary_address.postcode',
        'primary_email.address',
        'primary_phone.number',
        'receives_district_notifications',
        'receives_circuit_notifications',
    ];

    /**
     * The related Address models.
     *
     * @return \Illuminate\Database\Eloquent\Relations\MorphMany
     */
    public function addresses()
    {
        return $this->morphMany(Address::class, 'owner')->orderByDesc('primary');
    }

    /**
     * The related Phone models.
     *
     * @return \Illuminate\Database\Eloquent\Relations\MorphMany
     */
    public function phones()
    {
        return $this->morphMany(Phone::class, 'owner')->orderByDesc('primary');
    }

    /**
     * The related Email models.
     *
     * @return \Illuminate\Database\Eloquent\Relations\MorphMany
     */
    public function emails()
    {
        return $this->morphMany(Email::class, 'owner')->orderByDesc('primary');
    }

    /**
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    public function primary_address()
    {
        return $this->belongsTo(Address::class);
    }

    /**
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    public function primary_email()
    {
        return $this->belongsTo(Email::class);
    }

    /**
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    public function primary_phone()
    {
        return $this->belongsTo(Phone::class);
    }

    /**
     * The related Church models.
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function churches()
    {
        return $this->hasMany(Church::class, 'minister_id');
    }

    /**
     * The related Circuit model.
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    public function circuit()
    {
        return $this->belongsTo(Circuit::class);
    }

    /**
     * The related Group model.
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    public function group()
    {
        return $this->belongsTo(Group::class);
    }

    /**
     * The related MemberType model.
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    public function type()
    {
        return $this->belongsTo(MemberType::class);
    }

    /**
     * The related User model.
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasOne
     */
    public function user()
    {
        return $this->hasOne(User::class);
    }

    /**
     * The related Groups models.
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsToMany
     */
    public function groups()
    {
        return $this->belongsToMany(Group::class, 'group_members')->withPivot('role_id')->withTimestamps();
    }

    /**
     * The name attribute.
     *
     * @return string
     */
    public function getNameAttribute()
    {
        return trim(sprintf('%s %s %s', trim($this->title), trim($this->first_name), trim($this->last_name)));
    }

    /**
     * Directory Role related methods
     */

    /**
     * @return \Illuminate\Database\Eloquent\Relations\BelongsToMany
     */
    public function roles()
    {
        return $this->belongsToMany(Role::class, 'directory_member_role')
                    ->withPivot('id', 'scope_id', 'scope_type')->using(MemberRolePivot::class)
                    ->withTimestamps();
    }

    /**
     * @return \Illuminate\Database\Eloquent\Collection
     */
    public function getDistrictRolesAttribute()
    {
        return $this->roles()->wherePivot('scope_type', District::class)->withPivot('description')->get();
    }

    /**
     * @return \Illuminate\Database\Eloquent\Collection
     */
    public function getCircuitRolesAttribute()
    {
        return $this->roles()->wherePivot('scope_type', Circuit::class)->withPivot('description')->get();
    }

    /**
     * @return \Illuminate\Database\Eloquent\Collection
     */
    public function getChurchRolesAttribute()
    {
        return $this->roles()->wherePivot('scope_type', Church::class)->withPivot('description')->get();
    }

    /**
     * Directory GDPR related methods
     */

    /**
     * The GDPR records for this member.
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function gdpr()
    {
        return $this->hasMany(GdprRecord::class);
    }

    /**
     * Get the last GDPR authorization date.
     *
     * @return \Carbon\Carbon|null
     */
    public function getGdprAuthorizedAtAttribute()
    {
        $gdpr = $this->gdpr->last();

        return is_null($gdpr) ? $gdpr : $gdpr->authorized_at;
    }

    /**
     * @param \Illuminate\Database\Eloquent\Builder $query
     *
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function scopeWithoutValidGdprRecord(Builder $query)
    {
        return $query->doesntHave('gdpr')->orWhereHas('gdpr', function (Builder $query) {
            return $query->whereDate('authorized_at', '<', Carbon::now()->subMonthNoOverflow(12));
        });
    }

    /**
     * @param Member|array $member
     * @param string       $message
     */
    public function sendPrivateMessage($member, $message)
    {
        $messenger = app()->make(Services\Messenger::class);

        if (is_array($member) || $member instanceof \IteratorAggregate) {
            foreach ($member as $m) {
                $this->sendPrivateMessage($m, $message);
            }
        }

        if (is_a($member, Member::class) && $member !== $this && !empty($member->primary_email->address)) {
            $messenger->message($this->name, $member->primary_email->address, $message);
        }
    }

    /**
     * Route notifications for the mail channel.
     *
     * @return string
     */
    public function routeNotificationForMail()
    {
        return $this->primary_email->address;
    }

    public function logActivity($eventName, $properties)
    {
        $description = $this->getDescriptionForEvent($eventName);

        $logName = $this->getLogNameToUse($eventName);

        if ($description == '' || count($properties) === 0) {
            return;
        }

        app(ActivityLogger::class)
            ->useLog($logName)
            ->performedOn($this)
            ->withProperties($properties)
            ->log($description);
    }
}
