<?php namespace DistrictHub\Directory\Http\Controllers;

use App\Http\Controllers\Controller;
use DistrictHub\Directory\Address;
use DistrictHub\Directory\AddressTitle;
use DistrictHub\Directory\AddressType;
use DistrictHub\Directory\Church;
use DistrictHub\Directory\Http\Requests;
use DistrictHub\Directory\Member;
use Illuminate\Http\Request;
use Laracasts\Flash\Flash;

class ChurchAddressController extends Controller
{
    /*
    |--------------------------------------------------------------------------
    | Directory Address Controller
    |--------------------------------------------------------------------------
    */

    /**
     * Create a new controller instance.
     *
     * @return void
     */
    public function __construct()
    {
        $this->middleware('auth');
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return Response
     */
    public function create(Request $request)
    {
        $route = $request->route();
        $memberId = $route->parameter('member');
        $churchId = $route->parameter('church');

        if (!is_null($memberId)) {
            $owner = Member::find($memberId);
        }

        if (!is_null($churchId)) {
            $owner = Church::find($churchId);
        }

        return view('directory::addresses.create', compact('owner'));
    }

    /**
     * Store a newly created resource in storage.
     *
     * @return Response
     */
    public function store(Requests\AddressRequest $request)
    {
        $address = Address::create($request->all());

        $route = $request->route();
        $memberId = $route->parameter('member');
        $churchId = $route->parameter('church');

        if (!is_null($memberId)) {
            $member = Member::find($memberId);
            $member->addresses()->save($address);

            $member->logActivity('updated', [
                'attributes' => [
                    'address.address_line1' => $address->address_line1,
                    'address.address_line2' => $address->address_line2,
                    'address.address_line3' => $address->address_line3,
                    'address.city'          => $address->city,
                    'address.postcode'      => $address->postcode,
                ],
                'old'        => [
                    'address.address_line1' => '',
                    'address.address_line2' => '',
                    'address.address_line3' => '',
                    'address.city'          => '',
                    'address.postcode'      => '',
                ],
            ]);

            if ($address->primary) {
                $member->primary_address_id = $address->id;
                $member->save();
            }
        }

        if (!is_null($churchId)) {
            $church = Church::find($churchId);
            $church->address()->save($address);
        }

        Flash::success("Created Address \"{$address->full_address}\"");

        return redirect()->back();
    }

    /**
     * Display the specified resource.
     *
     * @return Response
     */
    public function show(Request $request)
    {
        $addressId = $request->route()->parameter('address');

        $address = Address::find($addressId);

        return view('directory::addresses.show', compact('address'));
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @return Response
     */
    public function edit(Request $request)
    {
        $addressId = $request->route()->parameter('address');

        $address = Address::find($addressId);

        $memberId = $request->route()->parameter('member');
        $churchId = $request->route()->parameter('church');

        if (!is_null($memberId)) {
            $owner = Member::find($memberId);
        }

        if (!is_null($churchId)) {
            $owner = Church::find($churchId);
        }

        return view('directory::addresses.edit', compact('address', 'owner'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @return Response
     */
    public function update(Church $church, Address $address, Requests\AddressRequest $request)
    {
        $address->fill($request->all());
        $address->save();

        Flash::success("Updated Address \"{$address->full_address}\"");

        return redirect()->back();
    }

    /**
     * Show delete confirmation.
     *
     * @return Response
     */
    public function delete($memberId, $addressId)
    {
        return view('directory::addresses.delete', compact('memberId', 'addressId'));
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int $id
     *
     * @return Response
     */
    public function destroy(Member $member, $addressId)
    {
        $address = Address::find($addressId);

        Address::destroy($addressId);

        $member->logActivity('updated', [
            'attributes' => [
                'address.address_line1' => '',
                'address.address_line2' => '',
                'address.address_line3' => '',
                'address.city'          => '',
                'address.postcode'      => '',
            ],
            'old'        => [
                'address.address_line1' => $address->address_line1,
                'address.address_line2' => $address->address_line2,
                'address.address_line3' => $address->address_line3,
                'address.city'          => $address->city,
                'address.postcode'      => $address->postcode,
            ],
        ]);

        Flash::success("Deleted Address \"{$address->full_address}\"");

        return redirect()->back();
    }
}
