<?php namespace DistrictHub\Groups\Http\Controllers;

use DistrictHub\Groups\Jobs\CreateGroup;
use DistrictHub\Groups\Jobs\LinkMemberToGroup;
use DistrictHub\Groups\Jobs\UnlinkMemberFromGroup;
use DistrictHub\Groups\Jobs\UpdateGroup;
use DistrictHub\Groups\Group;
use DistrictHub\Groups\Role;
use App\Http\Controllers\Controller;
use DistrictHub\Groups\Http\Requests;
use Illuminate\Contracts\Auth\Authenticatable;
use Illuminate\Support\Facades\Auth;
use Laracasts\Flash\Flash;

class GroupController extends Controller
{
    /*
    |--------------------------------------------------------------------------
    | Group Controller
    |--------------------------------------------------------------------------
    */

    /**
     * Create a new controller instance.
     *
     * @return void
     */
    public function __construct()
    {
        $this->middleware('auth');
    }

    /**
     * Display a listing of the resource.
     *
     * @return Response
     */
    public function index()
    {
        $groups = Group::all();

        return view('groups::index', compact('groups'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return Response
     */
    public function create()
    {
        return view('groups::create');
    }

    /**
     * Store a newly created resource in storage.
     *
     * @return Response
     */
    public function store(Requests\GroupRequest $request)
    {
        $creatorMemberId = Auth::user()->member->id;

        $this->dispatch(new CreateGroup($request->input('name'), $request->input('description'), $creatorMemberId));

        return redirect(route('groups.index'));
    }

    /**
     * Display the specified resource.
     *
     * @param  int $id
     *
     * @return Response
     */
    public function show($id)
    {
        $group = Group::find($id);

        $members = $group->members()->whereIn('group_members.role_id', [1, 2])->orderBy('last_name', 'asc')->get();

        $roles = Role::all()->pluck('description', 'id');

        $posts = $group->posts()->orderBy('created_at', 'DESC')->paginate(5);

        return view('groups::show', compact('group', 'members', 'roles', 'posts'));
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int $id
     *
     * @return Response
     */
    public function edit($id)
    {
        if (Auth::user()->adminOf($id)) {
            $group = Group::find($id);

            $roles = Role::all()->pluck('description', 'id');

            return view('groups::edit', compact('group', 'roles'));
        }

        return \App::abort(403);
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  int $id
     *
     * @return Response
     */
    public function update($id, Requests\GroupRequest $request)
    {
        if (Auth::user()->adminOf($id)) {
            $name = $request->input('name');
            $description = $request->input('description');
            $contribute = (bool)$request->input('contribute');
            $membership_list = (bool)$request->input('membership_list');

            $this->dispatch(new UpdateGroup($id, $name, $description, $contribute, $membership_list));

            return redirect()->back();
        }

        return \App::abort(403);
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int $id
     *
     * @return Response
     */
    public function destroy($id)
    {
        $group = Group::findOrFail($id);

        $group->delete();

        Flash::success("Deleted \"{$group->name}\"");

        return redirect()->back();
    }

    public function delete($groupId)
    {
        return view('groups::delete', compact('groupId'));
    }

    public function compose($id)
    {
        return view('groups::compose', compact('id'));
    }

    public function message($id, Requests\MessageRequest $request)
    {
        $group = Group::where('id', $id)->firstOrFail();

        $roles = [];

        if($request->input('members')) {
            $roles[] = 1;
            $roles[] = 2;
        }

        if($request->input('subscribers')) {
            $roles[] = 3;
        }

        $members = $group->members()->whereIn('group_members.role_id', $roles)->get();

        Auth::user()->member->sendPrivateMessage($members, $request->input('message'));

        return redirect()->back();
    }

    /**
     * Link member to specified resource.
     *
     * @param int                        $id
     * @param Requests\LinkMemberRequest $request
     *
     * @return Response
     */
    public function linkMember($id, Requests\LinkMemberRequest $request)
    {
        $this->dispatch(new LinkMemberToGroup($request->input('member_id'), $id, $request->input('role_id')));

        return redirect()->back();
    }

    /**
     * Unlink member from specified resource.
     *
     * @param int $groupId
     * @param int $request
     *
     * @return Response
     */
    public function unlinkMember($groupId, $memberId)
    {
        $this->dispatch(new UnlinkMemberFromGroup($memberId, $groupId));

        return redirect()->back();
    }

    /**
     * Allow current member to join group.
     *
     * @param int             $groupId
     * @param Authenticatable $user
     *
     * @return Response
     */
    public function joinGroup($groupId, Authenticatable $user)
    {
        $this->dispatch(new LinkMemberToGroup($user->member_id, $groupId, 2));

        return redirect()->back();
    }

    /**
     * Allow current member to leave group.
     *
     * @param int             $groupId
     * @param Authenticatable $user
     *
     * @return \Illuminate\Http\RedirectResponse
     */
    public function leaveGroup($groupId, Authenticatable $user)
    {
        $this->dispatch(new UnlinkMemberFromGroup($user->member_id, $groupId));

        return redirect()->back();
    }

    /**
     * Allow current member to join group.
     *
     * @param int             $groupId
     * @param Authenticatable $user
     *
     * @return Response
     */
    public function subscribeToGroup($groupId, Authenticatable $user)
    {
        $this->dispatch(new LinkMemberToGroup($user->member_id, $groupId, 3));

        return redirect()->back();
    }
}
