<?php namespace DistrictHub\Groups;

use DistrictHub\Groups\Group;
use Carbon\Carbon;
use DistrictHub\Directory\Member;
use Illuminate\Database\Eloquent\Model;
use League\Flysystem\FileNotFoundException;
use Symfony\Component\HttpFoundation\File\UploadedFile;

class Post extends Model
{
    /**
     * The database table used by the model.
     *
     * @var string
     */
    protected $table = 'group_posts';

    /**
     * The attributes that are mass assignable.
     *
     * @var array
     */
    protected $fillable = [
        'author',
        'content',
        'attachment',
    ];

    /**
     * Related Group model.
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    public function group()
    {
        return $this->belongsTo(Group::class);
    }

    /**
     * Related Member model.
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    public function member()
    {
        return $this->belongsTo(Member::class);
    }

    /**
     * Get attachment attribute.
     *
     * @return Attachment
     */
    public function getAttachmentAttribute()
    {
        return new Attachment(
            'attachment',
            $this->attributes['attachment_file_name'],
            $this->attributes['attachment_file_size'],
            $this->attributes['attachment_content_type'],
            $this->attributes['attachment_updated_at']
        );
    }

    /**
     * Set attachment attribute
     *
     * @param mixed $value
     */
    public function setAttachmentAttribute($value)
    {
        if($value === '') {
            list($type, $subtype) = explode('/', $this->attributes['attachment_content_type']);

            try {
                \Storage::disk('s3')->delete([
                    sprintf('/%ss/%s/%s/%s', $type, 'attachment', 'original', $this->attributes['attachment_file_name']),
                    sprintf('/%ss/%s/%s/%s', $type, 'attachment', 'medium', $this->attributes['attachment_file_name']),
                    sprintf('/%ss/%s/%s/%s', $type, 'attachment', 'thumb', $this->attributes['attachment_file_name']),
                ]);
            } catch (FileNotFoundException $exception) {}

            $this->attributes['attachment_file_name'] = null;
            $this->attributes['attachment_file_size'] = null;
            $this->attributes['attachment_content_type'] = null;
            $this->attributes['attachment_updated_at'] = Carbon::now();
        }

        if($value instanceof UploadedFile) {
            $clientMimeType = $value->getClientMimeType();
            list($type, $subtype) = explode('/', $clientMimeType);

            $this->attributes['attachment_file_name'] = $value->getClientOriginalName();
            $this->attributes['attachment_file_size'] = $value->getClientSize();
            $this->attributes['attachment_content_type'] = $clientMimeType;
            $this->attributes['attachment_updated_at'] = Carbon::now();

            \Storage::disk('s3')->put(
                sprintf('/%ss/%s/%s/%s', $type, 'attachment', 'original', $this->attributes['attachment_file_name']),
                file_get_contents($value->getRealPath()),
                'public'
            );

            if ($type === 'image') {

                \Storage::disk('s3')->put(
                    sprintf('/%ss/%s/%s/%s', $type, 'attachment', 'medium', $this->attributes['attachment_file_name']),
                    (string) \Image::make($value->getRealPath())->fit(300, 300)->encode('jpg', 100),
                    'public'
                );

                \Storage::disk('s3')->put(
                    sprintf('/%ss/%s/%s/%s', $type, 'attachment', 'thumb', $this->attributes['attachment_file_name']),
                    (string) \Image::make($value->getRealPath())->fit(100, 100)->encode('jpg', 100),
                    'public'
                );
            }
        }
    }

    public function getAttachmentSizeAttribute()
    {
        return $this->formatBytes($this->attachment->size(), 0);
    }

    protected function formatBytes($bytes, $precision = 2) {
        $units = array('B', 'KB', 'MB', 'GB', 'TB');
        $factor = floor((strlen($bytes) - 1) / 3);
        return sprintf("%.{$precision}f", $bytes / pow(1024, $factor)) . @$units[$factor];
    }
}
