<?php

/**
 * ContactForm.
 *
 * @author       Tom Densham <tom.densham@studiobonito.co.uk>
 * @copyright    Studio Bonito Ltd.
 */
class ContactForm extends Form
{
    /**
     * Create a new form, with the given fields and action buttons.
     * Fallback to default fields and action buttons if none are supplied.
     *
     * @param Controller $controller
     * @param String     $name
     * @param FieldList  $fields
     * @param FieldList  $actions
     * @param Validator  $validator
     */
    public function __construct(
        $controller,
        $name,
        FieldList $fields = null,
        FieldList $actions = null,
        $validator = null
    ) {
        if (!$fields || !$fields instanceof FieldList) {
            $fields = $this->getDefaultFields();
        }
        if (!$actions || !$actions instanceof FieldList) {
            $actions = $this->getDefaultActions();
        }
        if (!$validator || !$validator instanceof Validator) {
            $validator = $this->getDefaultValidator();
        }

        parent::__construct($controller, $name, $fields, $actions, $validator);
    }

    /**
     * Create the default fields.
     *
     * @return FieldList
     */
    public function getDefaultFields()
    {
        $fields = FieldList::create();

        $fields->push(TextField::create('Name', _t('CONTACTFORM.Name', 'Name:')));
        $fields->push(TextField::create('Email', _t('CONTACTFORM.Email', 'Email Address:')));
        $fields->push(TextField::create('Telephone', _t('CONTACTFORM.Telephone', 'Telephone Number:')));
        $fields->push(TextareaField::create('Message', _t('CONTACTFORM.Message', 'Please leave your comment below:')));

        $controller = Controller::curr();

        if ($controller->request->getVar('OfficerID')) {
            $officerID = HiddenField::create('OfficerID');
            $officerID->setValue($controller->request->getVar('OfficerID'));
            $fields->push($officerID);
        } else {
            if ($controller->request->getVar('MinisterName')) {
                $MinisterName = HiddenField::create('MinisterName');
                $MinisterName->setValue($controller->request->getVar('MinisterName'));
                $fields->push($MinisterName);
            }
        }

        return $fields;
    }

    /**
     * Create the default actions.
     *
     * @return FieldList
     */
    public function getDefaultActions()
    {
        $actions = FieldList::create();

        $searchAction = new FormAction('send', 'Submit');
        $searchAction->setUseButtonTag(true);
        $searchAction->addExtraClass('btn float--right');
        $actions->push($searchAction);

        return $actions;
    }

    /**
     * Create the default validator.
     *
     * @return Validator
     */
    public function getDefaultValidator()
    {
        $validator = ZenValidator::create();

        $validator->addRequiredFields(
            [
                'Name'    => 'Please enter your name',
                'Email'   => 'Please enter your email address',
                'Message' => 'Please add a short message outlining your request for contact',
            ]
        );

        $validator->setConstraint('Email', Constraint_type::create('email'));

        return $validator;
    }

    public function send(array $data, Form $form)
    {
        $controller = Controller::curr();

        $toEmail = $controller->getField('ToEmail');

        if (array_key_exists('OfficerID', $data)) {
            $officer = DistrictOfficer::get()->byId($data['OfficerID']);
            $toEmail = $officer->EmailAddress;
        }

        if (array_key_exists('MinisterName', $data)) {
            $config = SiteConfig::current_site_config();

            $service = new RestfulService($config->SearchApiUrl);
            $response = $service->request('/api/v1/ministers?exact=true&name=' . $data['MinisterName']);

            $json = Convert::json2array($response->getBody());

            $minister = $json['data'][0];
            $toEmail = $minister['email_address'];
        }

        if ($toEmail) {
            $email = new Email();
            $email
                ->setFrom($data['Email'])
                ->setTo($toEmail)
                ->setSubject($data['Name'] . ' would like to get in contact.')
                ->setTemplate('ContactFormEmail')
                ->populateTemplate($data);
            $email->send();

            if (array_key_exists('OfficerID', $data)) {
                $form->sessionMessage(
                    'Your message has been sent to ' . $officer->Name . '.',
                    'success'
                );

                return $this->controller->redirect($controller->Link());
            }

            $form->sessionMessage(
                'Your message has been sent.',
                'success'
            );
        }

        return $this->controller->redirect($controller->Link());
    }
}
