<?php

/**
 * Page type that provides overview and content for an event.
 *
 * @author       Steve Heyes <steve.heyes@studiobonito.co.uk>
 * @copyright    Studio Bonito Ltd.
 */
class EventPage extends Page
{
    /**
     * Human-readable singular name.
     * @var string
     * @config
     */
    private static $singular_name = 'Event';

    /**
     * Human-readable plural name
     * @var string
     * @config
     */
    private static $plural_name = 'Events';

    /**
     * List of database fields. {@link DataObject::$db}
     *
     * @var array
     */
    private static $db = array(
        'Summary'               => 'HTMLText',
        'StartDate'             => 'SS_Datetime',
        'EndDate'               => 'SS_Datetime',
        'ExternalLink'          => 'Varchar(2083)',
        'OrganiserName'         => 'Varchar(128)',
        'OrganiserTelephone'    => 'Varchar(128)',
        'OrganiserEmail'        => 'Varchar(128)',
        'PromotedEvent'         => 'Boolean'
    );


    /**
     * List of one-to-many relationships. {@link DataObject::$has_many}
     *
     * @var array
     */
    private static $has_one = array(
        'Type' => 'Type',
        'Theme' => 'Theme'
    );

    /**
     * List of many-to-many relationships. {@link DataObject::$many_many}
     *
     * @var array
     */
    private static $many_many = array(
        'RelatedEvents' => 'EventPage'
    );

    public static $many_many_extraFields=array(
        'RelatedEvents' => array(
            'SortOrder'=>'Int'
        )
    );

    private static $searchable_fields = array(
        'Title',
        'Summary',
        'Content'
    );

    /**
     * List of fields used as columns in the CMS.
     *
     * @var array
     */
    private static $summary_fields = array(
        'Title'      => 'Title',
        'StartDate' => 'Start Date',
        'EndDate' => 'End Date'
    );

    static $defaults = array (
        'ShowInMenus' => false
    );

    /**
     * Don't return the end date if it's the same as the start date.
     *
     * @return mixed|null
     */
    public function getEndDate()
    {
        $endDate = $this->getField('EndDate');

        if ($endDate === $this->StartDate) {
            return null;
        }

        return $endDate;
    }

    /**
     * Returns a FieldList with which to create the main editing form. {@link DataObject::getCMSFields()}
     *
     * @return FieldList The fields to be displayed in the CMS.
     */
    public function getCMSFields()
    {
        $fields = parent::getCMSFields();

        $fields->removeByName('MenuTitle');

        if ($this->ID === 0) {
            $fields->removeByName('URLSegment');
        }

        // Add Before Content
        $fields->addFieldsToTab(
            'Root.Main',
            array(
                HtmlEditorField::create('Summary')->setRows(15)
            ),
            'Content'
        );

        // Add After Content
        $fields->addFieldsToTab(
            'Root.Main',
            array(
                TextField::create('ExternalLink', _t('EventPage.EXTERNALLINK', 'External Link')),
                CheckboxField::create('PromotedEvent', _t('EventPage.PROMOTEDEVENT', 'Promoted Event')),
            ),
            'Metadata'
        );

        $startDateField = DatetimeField::create('StartDate', _t('EventPage.STARTDATE', 'Start Date'));
        $startDateField->getDateField()->setConfig('showcalendar', true);

        $endDateField = DatetimeField::create('EndDate', _t('EventPage.ENDDATE', 'End Date'));
        $endDateField->getDateField()->setConfig('showcalendar', true);

        // Add Date Information
        $fields->addFieldsToTab(
            'Root.Dates',
            array(
                $startDateField,
                $endDateField
            )
        );

        $typeDropdownField = $this->createDropdownField('Type');
        $themeDropdownField = $this->createDropdownField('Theme');

        $fields->addFieldsToTab(
            'Root.Taxonomy',
            array(
                $typeDropdownField,
                $themeDropdownField
            )
        );

        $fields->addFieldsToTab(
            'Root.Organiser',
            array(
                TextField::create('OrganiserName', _t('EventPage.ORGANISERNAME', 'Organiser Name')),
                TextField::create('OrganiserTelephone', _t('EventPage.ORGANISERNAME', 'Organiser Telephone')),
                TextField::create('OrganiserEmail', _t('EventPage.ORGANISERNAME', 'Organiser Email'))
            )
        );

        $conf = GridFieldConfig_RelationEditor::create(10);
        $conf->addComponent(new GridFieldOrderableRows('SortOrder'));

        $gridfield = GridField::create('RelatedEvents', 'Related Events', $this->RelatedEvents(), $conf);

        $fields->addFieldToTab('Root.RelatedEvents', $gridfield);


        return $fields;
    }

    public function RelatedEvents() {
        return $this->getManyManyComponents('RelatedEvents')->sort('SortOrder');
    }

    /**
     * Set the end date to the same as the start date if empty.
     */
    protected function onBeforeWrite()
    {
        parent::onBeforeWrite();

        if (is_null($this->EndDate) && !is_null($this->StartDate)) {
            $this->EndDate = $this->StartDate;
        }
    }

    /**
     * Creates a DropdownField given a field name and object class.
     *
     * @param      $name
     * @param null $className
     *
     * @return DropdownField
     * @throws BadFunctionCallException
     */
    protected function createDropdownField($name, $className = null)
    {
        if (is_null($className)) {
            if (class_exists($name)) {
                $className = $name;
            } else {
                throw new BadFunctionCallException('Either $className must be provided or $name must be a valid class.');
            }
        }

        $source = function () use ($className) {
            return $className::get()->map('ID', 'Title');
        };

        $dropdownField = DropdownField::create(
            $name."ID",
            $this->fieldLabel($name),
            $source()
        );
        $dropdownField->useAddNew($className, $source)
            ->setEmptyString('(Select one '.$name.')');

        return $dropdownField;


    }
}

/**
 * Controller that provides standard functions.
 *
 * @author       Steve Heyes <steve.heyes@studiobonito.co.uk>
 * @copyright    Studio Bonito Ltd.
 */
class EventPage_Controller extends Page_Controller
{

}