<?php

/**
 * BlogTag.
 *
 * @author       Tom Densham <tom.densham@studiobonito.co.uk>
 * @copyright    Studio Bonito Ltd.
 */
class BlogTag extends DataObject
{
    /**
     * List of database fields. {@link DataObject::$db}
     *
     * @var array
     */
    private static $db = array(
        'Title'      => 'Varchar(255)',
        'URLSegment' => 'Varchar(255)',
    );

    /**
     * List of one-to-one relationships. {@link DataObject::$has_one}
     *
     * @var array
     */
    private static $has_one = array(
        'Blog' => 'Blog',
    );

    /**
     * List of many-to-many relationships. {@link DataObject::$many_many}
     *
     * @var array
     */
    private static $belongs_many_many = array(
        'BlogPosts' => 'BlogPost',
    );

    /**
     * Returns a FieldList with which to create the main editing form.
     *
     * @return FieldList
     */
    public function getCMSFields()
    {
        $fields = parent::getCMSFields();

        $fields->removeByName('BlogID');

        $fields->removeByName('BlogPosts');

        return $fields;
    }

    /**
     * @return bool
     */
    public function Current()
    {
        $url = Controller::curr()->getRequest()->getURL();
        $link = trim($this->Link(), '/');

        if ($url === $link) {
            return true;
        }

        return false;
    }

    /**
     * @return string
     */
    public function Link()
    {
        $blog = $this->Blog();

        if ($blog instanceof Blog) {
            return $blog->Link(Controller::join_links($blog->TagBaseURLSegment, $this->URLSegment));
        }

        return null;
    }

    /**
     * @return FieldList
     */
    public function getAddNewFields()
    {
        $fields = new FieldList();

        $fields->push(new TextField('Title'));

        $fields->push(new TextField('URLSegment'));

        return $fields;
    }

    /**
     *
     */
    public function onBeforeWrite()
    {
        parent::onBeforeWrite();

        $this->generateURLSegment();
    }

    /**
     * Generates a unique URLSegment from the title.
     *
     * @param $increment
     *
     * @return string URLSegment
     */
    private function generateURLSegment($increment = null)
    {
        $filter = new URLSegmentFilter();
        $this->URLSegment = $filter->filter($this->Title);
        if (is_int($increment)) {
            $this->URLSegment .= '-' . $increment;
        }

        // Check to see if the URLSegment already exists
        $duplicate = new DataList($this->ClassName);

        $duplicate = $duplicate->filter(
            array(
                'URLSegment' => $this->URLSegment,
                'BlogID'     => $this->BlogID,
            )
        );

        if ($this->ID) {
            $duplicate = $duplicate->exclude('ID', $this->ID);
        }

        if ($duplicate->count() > 0) {
            $increment = is_int($increment) ? $increment + 1 : 0;
            $this->generateURLSegment((int) $increment);
        }

        return $this->URLSegment;
    }
}
