<?php

/**
 * Blog page is a container for blog post pages.
 *
 * @author       Tom Densham <tom.densham@studiobonito.co.uk>
 * @copyright    Studio Bonito Ltd.
 */
class Blog extends Page
{
    /**
     * Indicates what kind of children this page type can have. {@link SiteTree::$allowed_children}
     *
     * @config
     * @var array
     */
    private static $allowed_children = array(
        'BlogPost'
    );

    /**
     * An array of extension names and parameters to be applied to this object upon construction. {@link Object::$extensions}
     *
     * @config
     * @var array
     */
    private static $extensions = array(
        'BlogSiteTreeFilterExtension'
    );

    /**
     * The default child class for this page. {@link SiteTree::$default_child}
     *
     * @config
     * @var string
     */
    private static $default_child = 'BlogPost';

    /**
     * Display blog posts with published dates in the future.
     *
     * @config
     * @var bool
     */
    private static $display_future_posts = false;

    /**
     * Returns a list of all {@link BlogPost}s that are children of this Blog.
     *
     * @return DataList
     */
    public function getBlogPosts($sort = 'DESC')
    {
        $date = SS_Datetime::now();

        $posts = BlogPost::get()->filter(array('ParentID' => $this->ID))->sort('Published', $sort);

        if ($this->config()->get('display_future_posts') === false) {
            $posts = $posts->exclude(array('Published:GreaterThan' => $date->getValue()));
        }

        $this->extend('updateBlogPosts', $posts);

        return $posts;
    }

    /**
     * Returns a list of all {@link BlogPost}s that are children of this Blog and
     * have a Published date that is within the $year, $month, $day parameters.
     *
     * @param      $year
     * @param null $month
     * @param null $day
     *
     * @return DataList
     */
    public function getArchivedBlogPosts($year, $month = null, $day = null, $sort = 'DESC')
    {
        if ($day && $month) {
            $dayFormatted = str_pad($day, 2, '0', STR_PAD_LEFT);

            $startDate = "$year-$month-$dayFormatted 00:00:00";
            $endDate = "$year-$month-$dayFormatted 23:59:59";
        } elseif ($month) {
            $maxDays = cal_days_in_month(CAL_GREGORIAN, $month, $year);

            $startDate = "$year-$month-01 00:00:00";
            $endDate = "$year-$month-$maxDays 00:00:00";
        } else {
            $startDate = "$year-01-01 00:00:00";
            $endDate = "$year-12-31 00:00:00";
        }

        $posts = $this->getBlogPosts()
            ->filter(
                array(
                    'Published:GreaterThanOrEqual' => $startDate,
                    'Published:LessThanOrEqual'    => $endDate
                )
            )->sort('Published', $sort);

        return $posts;
    }

    /**
     * Returns a FieldList with which to create the main editing form. {@link DataObject::getCMSFields()}
     *
     * @codeCoverageIgnore
     *
     * @return FieldList The fields to be displayed in the CMS.
     */
    public function getCMSFields()
    {
        $fields = parent::getCMSFields();

        if (count($this->getExcludedSiteTreeClassNames()) > 0) {
            $gridFieldConfig = GridFieldConfig_RecordEditor::create()
                ->removeComponentsByType('GridFieldAddNewButton')
                ->addComponent(new GridFieldSiteTreeAddNewButton('buttons-before-left'), 'GridFieldToolbarHeader')
                ->removeComponentsByType('GridFieldEditButton')
                ->addComponent(new GridFieldSiteTreeEditButton(), 'GridFieldDeleteAction');

            $fields->addFieldsToTab(
                'Root.BlogPosts',
                array(
                    GridField::create('BlogPosts', _t('Blog.BLOGPOSTS', 'Blog posts'), $this->getBlogPosts())
                        ->setConfig($gridFieldConfig)
                )
            );
        }

        return $fields;
    }

    /**
     * Loops through subclasses of {@link BlogPost} and checks whether they have been configured
     * to be hidden. If so, then they will be excluded from the {@link SiteTree}.
     *
     * @return array
     */
    public function getExcludedSiteTreeClassNames()
    {
        $classes = array();
        $allowedChildren = $this->allowedChildren();
        foreach ($allowedChildren as $class) {
            if (!Config::inst()->get($class, 'show_in_sitetree')) {
                $classes[$class] = $class;
            }
        }

        return $classes;
    }
}

/**
 * Controller for blog page.
 *
 * @author       Tom Densham <tom.densham@studiobonito.co.uk>
 * @copyright    Studio Bonito Ltd.
 */
class Blog_Controller extends Page_Controller
{
    /**
     * @var DataList
     */
    protected $blogPosts;

    private static $allowed_actions = array(
        'rss',
        'archive'
    );

    public function init()
    {
        parent::init();

        if($this->request->param('Action') == 'archive')
        {
            $year = $this->getYear();
            $this->blogPosts = $this->data()->getArchivedBlogPosts($year);
        } else {
            // If no {@link BlogPost}s are assigned retrieve the children of this Blog.
            if (!$this->blogPosts) $this->blogPosts = $this->data()->getBlogPosts();
        }

        RSSFeed::linkToFeed($this->Link() . "rss", _t("Blog.RSSTITLE", "10 Most Recently Blog Posts"));
    }

    public function getYear()
    {
        return $this->request->getVar('year');
    }

    /**
     * @return DataList
     */
    public function getBlogPosts()
    {
        return $this->blogPosts;
    }

    /**
     * @param DataList $blogPosts
     */
    public function setBlogPosts($blogPosts)
    {
        $this->blogPosts = $blogPosts;
    }

    /**
     * Returns a PaginatedList of all the {@link BlogPost}s for this Blog.
     *
     * @return PaginatedList|null
     */
    public function BlogPosts()
    {
        if (!$this->blogPosts) return null;

        $posts = PaginatedList::create($this->blogPosts);

        $start = $this->request->getVar($posts->getPaginationGetVar());
        $posts->setPageStart($start);

        return $posts;
    }

    public function rss()
    {
        $rss = new RSSFeed(
            $this->BlogPosts(),
            $this->Link(),
            _t("Blog.RSSTITLE", "10 Most Recently Blog Posts"),
            _t("Blog.RSSDESCRIPTION", "Shows a list of the 10 most recently published blog posts.")
        );

        return $rss->outputToBrowser();
    }

    public function BlogYears()
    {
        $query = new SQLQuery();
        $query->setFrom('BlogPost_Live');
        $query->setDistinct(true);
        $query->selectField('YEAR(Published)', 'Published');
        $years = $query->execute();

        $yearsArray = new ArrayList();
        $thisYear = $this->getYear();

        foreach($years as $year) {
            if($thisYear == $year['Published'])
            {
                $year['Selected'] = 'Selected';
            }
            $yearsArray->push($year);
        }

        return $yearsArray;
    }
}