<?php namespace StudioBonito\SilverStripe\Dashboard;

use Aws\Ses\SesClient;
use DashboardChart;
use DropdownField;

/**
 * Single sentence description.
 *
 * @author       Tom Densham <tom.densham@studiobonito.co.uk>
 * @copyright    Studio Bonito Ltd.
 */
class DashboardSesMetricsPanel extends \DashboardPanel
{
    /**
     * List of database fields. {@link DataObject::$db}
     *
     * @var array
     */
    private static $db = array(
        'DateFormat' => "Enum('mdy,dmy','dmy')",
        'DateRange'  => "Enum('week,month,year','month')",
        'MetricType' => "Enum('Complaints,Rejects,Bounces,DeliveryAttempts','DeliveryAttempts')",
    );

    /**
     * @var string The name of the template used for the contents of this panel.
     */
    protected $template = 'DashboardSesMetricsPanel';

    /**
     * Provide a title to the panel selection window.
     *
     * @return string
     */
    public function getLabel()
    {
        return _t('DashboardExtras.SES_METRICS', 'Amazon SES Metrics');
    }

    /**
     * Provide a description to the panel selection window.
     *
     * @return string
     */
    public function getDescription()
    {
        return _t('DashboardExtras.SES_METRICS_DESC', 'Displays a chart of Amazon SES metrics');
    }

    /**
     * Gets the {@link FieldList} object that is used to configure the fields on this panel.
     * Similar to getCMSFields().
     *
     * @return \FieldList
     */
    public function getConfiguration()
    {
        $fields = parent::getConfiguration();

        $fields->push(DropdownField::create("DateFormat", _t('Dashboard.DATEFORMAT', 'Date format'), array(
            'dmy' => date('j M, Y'),
            'mdy' => date('M j, Y')
        ))
            ->addExtraClass('no-chzn')
        );
        $fields->push(DropdownField::create("DateRange", _t('Dashboard.DATERANGE', 'Date range'), array(
            'week'  => _t('Dashboard.PREVIOUSSEVENDAYS', '7 days'),
            'month' => _t('Dashboard.PREVIOUSTHIRTYDAYS', '30 days'),
            'year'  => _t('Dashboard.PREVIOUSYEAR', '365 days')
        ))
            ->addExtraClass('no-chzn')
        );
        $fields->push(DropdownField::create("MetricType", _t('Dashboard.METRICTYPE', 'Metric type'),
            $this->getMetricTypeLabels()
        )
            ->addExtraClass('no-chzn')
        );

        return $fields;
    }

    /**
     * Gets a timestamp for the start date of reporting, based on user provided data.
     *
     * @return int
     */
    public function getStartDateStamp()
    {
        switch ($this->DateRange) {
            case "day":
                return time();
            case "month":
                return strtotime("-30 days");
            case "year":
                return strtotime("-1 year");
            default:
                return strtotime("-30 days");
        }
    }

    /**
     * Gets a format string for date time.
     *
     * @return string
     */
    public function getDateFormatString()
    {
        switch ($this->DateFormat) {
            case 'mdy':
                return 'M j, Y';
            default:
                return 'j M, Y';
        }
    }

    /**
     * Configure chart for dashboard.
     *
     * @return DashboardChart
     */
    public function Chart()
    {
        $fromDate = date($this->DateFormatString, $this->StartDateStamp);
        $toDate = date($this->DateFormatString);

        $chart = DashboardChart::create(
            "{$fromDate} - {$toDate}",
            'Date',
            // Provide default here to ensure we always get a string from `getMetricTypeLabels`
            $this->getMetricTypeLabels($this->MetricType ?: 'DeliveryAttempts')
        );

        // If the panel hasn't been saved then no settings will exist so skip
        if ($this->MetricType) {
            $config = self::config()->get('config');

            $this->client = SesClient::factory($config);

            $response = $this->client->getSendStatistics(array());

            $data = $response->get('SendDataPoints');

            foreach ($data as $points) {
                $timestamp = strtotime($points['Timestamp']);

                if ($timestamp > $this->StartDateTimestamp && $timestamp < time()) {
                    $chart->addData(date($this->DateFormatString, $timestamp), $points[$this->MetricType]);
                }
            }
        }

        return $chart;
    }

    /**
     * Provide translations of enumeration options for metric types. Returns list
     * of options and translations or specific translation from a given option.
     *
     * @return array|string
     */
    private function getMetricTypeLabels($type = null)
    {
        $types = array(
            'Complaints'       => _t('Dashboard.COMPLAINTS', 'Complaints'),
            'Rejects'          => _t('Dashboard.REJECTS', 'Rejects'),
            'Bounces'          => _t('Dashboard.BOUNCES', 'Bounces'),
            'DeliveryAttempts' => _t('Dashboard.DELIVERYATTEMPTS', 'Deliveries'),
        );

        if ($type && isset($types[$type])) {
            return $types[$type];
        }

        return $types;
    }
}
