<?php

/**
 * Team stores a list of Team Members.
 * 
 * @author Tom Densham <tom.densham@studiobonito.co.uk>
 * @copyright (c) 2012, Studio Bonito Ltd.
 * @version 1.0
 */
class Team extends DataObject implements PermissionProvider
{
    /**
     * List of database fields. {@link DataObject::$db}
     * 
     * @access public
     * @static
     * @var array
     */
    public static $db = array(
        'Name' => 'Varchar(128)',
    );

    /**
     * List of many-to-many relationships. {@link DataObject::$many_many}
     * 
     * @access public
     * @static
     * @var array
     */
    public static $many_many = array(
        'Members' => 'TeamMember',
    );

    public static $summary_fields = array(
        'Name'
    );

    /**
     * Returns a FieldList with which to create the editing form. {@link SiteTree::getCMSFields()}
     * 
     * @return FieldList
     */
    public function getCMSFields()
    {
        $fields = new FieldList();
        $fields->push(new TextField('Name', _t('TEAMS.Name', 'Name')));
        $fields->push(new GridField('Members', _t('TEAMS.Members', 'Team Members'), $this->Members(), GridFieldConfig_RelationEditor::create()));
        return $fields;
    }

    public function providePermissions()
    {
        return array(
            'CREATE_TEAM' => array(
                'name'        => _t('TEAMS.CREATE_TEAM_DESCRIPTION', 'Create a new Team'),
                'category'    => _t('TEAMS.TEAM_PERMISSIONS_CATEGORY', 'Global Team permissions'),
                'sort'        => -80,
            //'help' => TODO: Add help text
            ),
            'DELETE_TEAM' => array(
                'name'      => _t('TEAMS.DELETE_TEAM_DESCRIPTION', 'Delete any Team'),
                'category'  => _t('TEAMS.TEAM_PERMISSIONS_CATEGORY', 'Global Team permissions'),
                'sort'      => -80,
            //'help' => TODO: Add help text
            ),
            'EDIT_TEAM' => array(
                'name'      => _t('TEAMS.EDIT_TEAM_DESCRIPTION', 'Edit any Team'),
                'category'  => _t('TEAMS.TEAM_PERMISSIONS_CATEGORY', 'Global Team permissions'),
                'sort'      => -80,
            //'help' => TODO: Add help text
            ),
            'VIEW_TEAM' => array(
                'name'     => _t('TEAMS.VIEW_TEAM_DESCRIPTION', 'View any Team'),
                'category' => _t('TEAMS.TEAM_PERMISSIONS_CATEGORY', 'Global Team permissions'),
                'sort'     => -80,
            //'help' => TODO: Add help text
            )
        );
    }

    public function canCreate($member = null)
    {
        return Permission::check('CREATE_TEAM', 'any', $member);
    }

    public function canDelete($member = null)
    {
        return Permission::check('DELETE_TEAM', 'any', $member);
    }

    public function canEdit($member = null)
    {
        return Permission::check('EDIT_TEAM', 'any', $member);
    }

    public function canView($member = null)
    {
        return Permission::check('VIEW_TEAM', 'any', $member);
    }
}

/**
 * Team Page Extension adds a Team to a Page with a dropdown field for selection.
 * 
 * @author Tom Densham <tom.densham@studiobonito.co.uk>
 * @copyright (c) 2012, Studio Bonito Ltd.
 * @version 1.0
 */
class Team_PageExtension extends DataExtension
{
    /**
     * List of one-to-one relationships. {@link DataObject::$has_one}
     * 
     * @access public
     * @static
     * @var array
     */
    public static $has_one = array(
        'Team' => 'Team',
    );

    /**
     * Updates the FieldList with which to create the editing form. {@link SiteTree::getCMSFields()}
     * 
     * @param FieldList $$fields
     * @return void
     */
    public function updateCMSFields(FieldList $fields)
    {
        parent::updateCMSFields($fields);
        $fields->addFieldToTab('Root.Team', new GridField('Team', 'Team', Team::get(), GridFieldConfig_RecordEditor::create()));
    }
}