<?php namespace StudioBonito\SilverStripe\Teams\Models;

use Controller;
use EmailField;
use Permission;
use TextField;

/**
 * Team Member stores personal details and social media links.
 *
 * @author       Tom Densham <tom.densham@studiobonito.co.uk>
 * @copyright    Studio Bonito Ltd.
 */
class TeamMember extends \DataObject implements \PermissionProvider
{
    const REGEX_FACEBOOK = '/^(https?:\/\/)?(www.)?(facebook.com\/)([A-Za-z0-9.]+)/';
    const REGEX_TWITTER = '/^(https?:\/\/)?(www.)?(twitter.com\/)(#!\/)?([A-Za-z0-9.]+)/';
    const REGEX_WEBSITE = '/^(http|https):\/\/([A-Z0-9][A-Z0-9_-]*(?:.[A-Z0-9][A-Z0-9_-]*)+):?(d+)?\/?/i';

    /**
     * Human-readable singular name. See {@link DataObject::$singular_name}
     *
     * @var string
     */
    public static $singular_name = 'Team Member';

    /**
     * Human-readable pluaral name. See {@link DataObject::$plural_name}
     *
     * @var string
     */
    public static $plural_name = 'Team Members';

    /**
     * List of database fields. {@link DataObject::$db}
     *
     * @access public
     * @static
     * @var array
     */
    public static $db = array(
        'Name'               => 'Varchar(128)',
        'Role'               => 'Varchar(128)',
        'Description'        => 'Text',
        'Email'              => 'Varchar(128)',
        'LinkedinUrl'        => 'Varchar(128)',
        'FacebookUrl'        => 'Varchar(128)',
        'TwitterUrl'         => 'Varchar(128)',
        'PersonalWebsiteUrl' => 'Varchar(128)',
    );

    /**
     * List of one-to-one relationships. {@link DataObject::$has_one}
     *
     * @access public
     * @static
     * @var array
     */
    public static $has_one = array(
        'Picture' => 'Image',
    );

    public static $summary_fields = array(
        'Name',
        'Email',
    );

    protected static $facebookPrefix = 'https://www.facebook.com/';

    protected static $twitterPrefix = 'https://twitter.com/';

    /**
     * If Facebook ID exists return the absolute url for the profile.
     *
     * @return string
     */
    public function getFacebookId()
    {
        $facebookUrl = $this->getField('FacebookUrl');

        return !empty($facebookUrl) ? self::$facebookPrefix . $facebookUrl : null;
    }

    /**
     * If Twitter ID exists return the absolute url for the profile.
     *
     * @return string
     */
    public function getTwitterId()
    {
        $twitterUrl = $this->getField('TwitterUrl');

        return !empty($twitterUrl) ? self::$twitterPrefix . $twitterUrl : null;
    }

    public function getMenuTitle()
    {
        return $this->Title;
    }

    public function getLink()
    {
        $teamHolder = TeamPage::get()->first();

        if ($teamHolder && $teamHolder->exists()) {
            $baseURL = $teamHolder->Link();
        } else {
            $baseURL = Controller::curr()->Link();
        }

        return Controller::join_links(
            $baseURL,
            'member',
            $this->Title
        );
    }

    public function getLinkingMode()
    {
        if (Controller::curr()->getAction() == 'member') {
            if (Controller::curr()->request->param('ID') == $this->Title) {
                return 'current';
            }
        }

        return 'link';
    }

    /**
     * Returns a FieldList with which to create the editing form. {@link SiteTree::getCMSFields()}
     *
     * @return FieldList
     */
    public function getCMSFields()
    {
        $fields = parent::getCMSFields();

        // Email field
        $emailField = new EmailField('Email', _t('TeamMember.EMAIL', 'Email'));
        $fields->replaceField('Email', $emailField);

        // LinkedIn Url field
        $linkedinUrlField = new TextField('LinkedinUrl', _t('TeamMember.LINKEDIN', 'LinkedIn'));
        $fields->replaceField('LinkedinUrl', $linkedinUrlField);

        // Facebook ID field
        $facebookIdField = new TextField('FacebookUrl', _t('TeamMember.FACEBOOK', 'Facebook'));
        $facebookIdField->setDescription('e.g. https://www.facebook.com/yourname');
        $fields->replaceField('FacebookUrl', $facebookIdField);

        // Twitter ID field
        $twitterIdField = new TextField('TwitterUrl', _t('TeamMember.TWITTER', 'Twitter'));
        $twitterIdField->setDescription('e.g. https://twitter.com/yourname');
        $fields->replaceField('TwitterUrl', $twitterIdField);

        // Personal Website Url field
        $personalWebsiteUrl = new TextField('PersonalWebsiteUrl', _t('TeamMember.PERSONALWEBSITE', 'Personal Website'));
        $personalWebsiteUrl->setDescription('e.g. http://www.example.com');
        $fields->replaceField('PersonalWebsiteUrl', $personalWebsiteUrl);

        return $fields;
    }

    public function validate()
    {
        $result = parent::validate();

        // Check if Facebook url is valid
        if (!empty($this->FacebookUrl) && preg_match(self::REGEX_FACEBOOK, $this->FacebookUrl) == false) {
            $result->error('Facebook is not a valid url');
        }

        // Check if Twitter url is valid
        if (!empty($this->TwitterUrl) && preg_match(self::REGEX_TWITTER, $this->TwitterUrl) == false) {
            $result->error('Twitter is not a valid url');
        }

        // Check if Personal Website url is valid
        if (!empty($this->PersonalWebsiteUrl) && preg_match(self::REGEX_WEBSITE, $this->PersonalWebsiteUrl) == false) {
            $result->error('Personal Website is not a valid url');
        }

        return $result;
    }

    public function providePermissions()
    {
        return array(
            'CREATE_TEAM_MEMBER' => array(
                'name'     => _t('Teams.CREATE_TEAM_MEMBER_DESCRIPTION', 'Create a new Team Member'),
                'category' => _t('Teams.TEAM_MEMBER_PERMISSIONS_CATEGORY', 'Global Team Member permissions'),
                'sort'     => -90,
                //'help' => TODO: Add help text
            ),
            'DELETE_TEAM_MEMBER' => array(
                'name'     => _t('Teams.DELETE_TEAM_MEMBER_DESCRIPTION', 'Delete any Team Member'),
                'category' => _t('Teams.TEAM_MEMBER_PERMISSIONS_CATEGORY', 'Global Team Member permissions'),
                'sort'     => -90,
                //'help' => TODO: Add help text
            ),
            'EDIT_TEAM_MEMBER'   => array(
                'name'     => _t('Teams.EDIT_TEAM_MEMBER_DESCRIPTION', 'Edit any Team Member'),
                'category' => _t('Teams.TEAM_MEMBER_PERMISSIONS_CATEGORY', 'Global Team Member permissions'),
                'sort'     => -90,
                //'help' => TODO: Add help text
            ),
            'VIEW_TEAM_MEMBER'   => array(
                'name'     => _t('Teams.VIEW_TEAM_MEMBER_DESCRIPTION', 'View any Team Member'),
                'category' => _t('Teams.TEAM_MEMBER_PERMISSIONS_CATEGORY', 'Global Team Member permissions'),
                'sort'     => -90,
                //'help' => TODO: Add help text
            ),
        );
    }

    public function canCreate($member = null)
    {
        return Permission::check('CREATE_TEAM_MEMBER', 'any', $member);
    }

    public function canDelete($member = null)
    {
        return Permission::check('DELETE_TEAM_MEMBER', 'any', $member);
    }

    public function canEdit($member = null)
    {
        return Permission::check('EDIT_TEAM_MEMBER', 'any', $member);
    }

    public function canView($member = null)
    {
        //return Permission::check('VIEW_TEAM_MEMBER', 'any', $member);
        return true;
    }

    /**
     * Ensure safe urls are stored for Facebook/Twitter.
     *
     * @return void
     */
    protected function onBeforeWrite()
    {
        parent::onBeforeWrite();

        if ($this->FacebookUrl) {
            // Sanitise the FacebookUrl
            preg_match(
                self::REGEX_FACEBOOK,
                $this->getField('FacebookUrl'),
                $matches
            );
            $this->FacebookUrl = self::$facebookPrefix . array_pop($matches);
        }

        if ($this->TwitterUrl) {
            // Sanitise the TwitterUrl
            preg_match(
                self::REGEX_TWITTER,
                $this->getField('TwitterUrl'),
                $matches
            );
            $this->TwitterUrl = self::$twitterPrefix . array_pop($matches);
        }
    }
}
