<?php

/**
 * Team Member stores personal details and social media links.
 * 
 * @author Tom Densham <tom.densham@studiobonito.co.uk>
 * @copyright (c) 2012, Studio Bonito Ltd.
 * @version 1.0
 */
class TeamMember extends DataObject implements PermissionProvider
{
    protected static $facebookPrefix = 'http://www.facebook.com/';

    protected static $twitterPrefix = 'http://www.twitter.com/';

    /**
     * List of database fields. {@link DataObject::$db}
     * 
     * @access public
     * @static
     * @var array
     */
    public static $db = array(
        'Name'               => 'Varchar(128)',
        'Description'        => 'Text',
        'Email'              => 'Varchar(128)',
        'LinkedinUrl'        => 'Varchar(128)',
        'FacebookId'         => 'Varchar(128)',
        'TwitterId'          => 'Varchar(128)',
        'PersonalWebsiteUrl' => 'Varchar(128)',
    );

    /**
     * List of one-to-one relationships. {@link DataObject::$has_one}
     * 
     * @access public
     * @static
     * @var array
     */
    public static $has_one = array(
        'Picture' => 'Image',
    );

    public static $belongs_many_many = array(
        'Teams' => 'Team',
    );

    public static $summary_fields = array(
        'Name',
        'Email'
    );

    /**
     * If Facebook ID exists return the absolute url for the profile.
     * 
     * @return string|boolean
     */
    public function getFacebookId()
    {
        $facebookId = $this->getField('FacebookId');
        return !empty($facebookId) ? self::$facebookPrefix.$facebookId : false;
    }

    /**
     * Get the absolute url for the Facebook profile.
     * 
     * @return string
     * 
     * @deprecated since version 2.0
     */
    public function getFacebookUrl()
    {
        return $this->getFacebookId();
    }

    /**
     * If Twitter ID exists return the absolute url for the profile.
     * 
     * @return string|boolean
     */
    public function getTwitterId()
    {
        $twitterId = $this->getField('TwitterId');
        return !empty($twitterId) ? self::$twitterPrefix.$twitterId : false;
    }

    /**
     * Get the absolute url for the Twitter profile.
     * 
     * @return type
     * 
     * @deprecated since version 2.0
     */
    public function getTwitterUrl()
    {
        return $this->getTwitterId();
    }

    /**
     * Returns a FieldList with which to create the editing form. {@link SiteTree::getCMSFields()}
     * 
     * @return FieldList
     */
    public function getCMSFields()
    {
        $fields = parent::getCMSFields();

        // Email field
        $emailField = new EmailField('Email', _t('TEAMS.Email', 'Email'));
        $fields->replaceField('Email', $emailField);

        // LinkedIn Url field
        $linkedinUrlField = new TextField('LinkedinUrl', _t('TEAMS.Linkedin', 'LinkedIn'));
        $fields->replaceField('LinkedinUrl', $linkedinUrlField);

        // Facebook ID field
        $facebookIdField = new TextField('FacebookId', _t('TEAMS.Facebook', 'Facebook'));
        $facebookIdField->setDescription('e.g. http://www.facebook.com/yourname');
        $fields->replaceField('FacebookId', $facebookIdField);

        // Twitter ID field
        $twitterIdField = new TextField('TwitterId', _t('TEAMS.Twitter', 'Twitter'));
        $twitterIdField->setDescription('e.g. http://www.twitter.com/yourname');
        $fields->replaceField('TwitterId', $twitterIdField);

        // Personal Website Url field
        $personalWebsiteUrl = new TextField('PersonalWebsiteUrl', _t('TEAMS.PersonalWebsite', 'Personal Website'));
        $personalWebsiteUrl->setDescription('e.g. http://www.example.com');
        $fields->replaceField('PersonalWebsiteUrl', $personalWebsiteUrl);

        return $fields;
    }

    public function validate()
    {
        $result = parent::validate();

        // Check if provided url is valid
        if (!empty($this->PersonalWebsiteUrl) && preg_match('/^(http|https):\/\/([A-Z0-9][A-Z0-9_-]*(?:.[A-Z0-9][A-Z0-9_-]*)+):?(d+)?\/?/i', $this->PersonalWebsiteUrl) == false) {
            $result->error('Personal Website is not a valid url');
        }

        return $result;
    }

    protected function onBeforeWrite()
    {
        parent::onBeforeWrite();

        // Remove facebook url and store only the facebook username
        preg_match('/^(https?:\/\/)?(www.)?(facebook.com\/)?([A-Za-z0-9.]+)/', $this->getField('FacebookId'), $matches);
        $this->FacebookId = array_pop($matches);

        // Remove twitter url and stroe only the twitter username
        preg_match('/^(https?:\/\/)?(www.)?(twitter.com\/)?(#!\/)?([A-Za-z0-9.]+)/', $this->getField('TwitterId'), $matches);
        $this->TwitterId = array_pop($matches);
    }
    
    public function providePermissions()
    {
        return array(
            'CREATE_TEAM_MEMBER' => array(
                'name'        => _t('TEAMS.CREATE_TEAM_MEMBER_DESCRIPTION', 'Create a new Team Member'),
                'category'    => _t('TEAMS.TEAM_MEMBER_PERMISSIONS_CATEGORY', 'Global Team Member permissions'),
                'sort'        => -90,
            //'help' => TODO: Add help text
            ),
            'DELETE_TEAM_MEMBER' => array(
                'name'      => _t('TEAMS.DELETE_TEAM_MEMBER_DESCRIPTION', 'Delete any Team Member'),
                'category'  => _t('TEAMS.TEAM_MEMBER_PERMISSIONS_CATEGORY', 'Global Team Member permissions'),
                'sort'      => -90,
            //'help' => TODO: Add help text
            ),
            'EDIT_TEAM_MEMBER' => array(
                'name'      => _t('TEAMS.EDIT_TEAM_MEMBER_DESCRIPTION', 'Edit any Team Member'),
                'category'  => _t('TEAMS.TEAM_MEMBER_PERMISSIONS_CATEGORY', 'Global Team Member permissions'),
                'sort'      => -90,
            //'help' => TODO: Add help text
            ),
            'VIEW_TEAM_MEMBER' => array(
                'name'     => _t('TEAMS.VIEW_TEAM_MEMBER_DESCRIPTION', 'View any Team Member'),
                'category' => _t('TEAMS.TEAM_MEMBER_PERMISSIONS_CATEGORY', 'Global Team Member permissions'),
                'sort'     => -90,
            //'help' => TODO: Add help text
            )
        );
    }

    public function canCreate($member = null)
    {
        return Permission::check('CREATE_TEAM_MEMBER', 'any', $member);
    }

    public function canDelete($member = null)
    {
        return Permission::check('DELETE_TEAM_MEMBER', 'any', $member);
    }

    public function canEdit($member = null)
    {
        return Permission::check('EDIT_TEAM_MEMBER', 'any', $member);
    }

    public function canView($member = null)
    {
        return Permission::check('VIEW_TEAM_MEMBER', 'any', $member);
    }
}
