<?php namespace StudioBonito\URLSegment\Extensions;

use Config;
use Controller;
use Convert;
use DataObject;
use Director;
use FieldList;
use ModelAsController;
use RootURLController;
use StudioBonito\URLSegment\Forms\URLSegmentField;
use URLSegmentFilter;

/**
 * URLSegmentExtension.
 *
 * @author       Tom Densham <tom.densham@studiobonito.co.uk>
 * @copyright    Studio Bonito Ltd.
 */
class URLSegmentExtension extends \DataExtension
{
    /**
     * List of database fields. {@link DataObject::$db}
     *
     * @var array
     */
    private static $db = [
        'URLSegment' => 'Varchar(255)',
    ];

    /**
     * List of labels for form fields.
     *
     * @var array
     */
    private static $field_labels = [
        'URLSegment' => 'URL Segment',
    ];

    /**
     * @return string
     */
    public function getTarget()
    {
        return Config::inst()->get($this->owner->class, 'urlsegment_target') ?: 'Title';
    }

    /**
     * Create and insert URLSegmentField to FieldList.
     *
     * @param FieldList $fields
     */
    public function updateCMSFields(FieldList $fields)
    {
        if($fields->dataFieldByName($this->getTarget())) {
            $baseLink = Controller::join_links(
                Director::absoluteBaseURL(),
                ($this->owner->ParentID ? $this->owner->Parent()->RelativeLink(true) : null)
            );

            $urlSegmentField = new URLSegmentField('URLSegment', $this->owner->fieldLabel('URLSegment'));
            $urlSegmentField->setTargetClass($this->owner->class);
            $urlSegmentField->setURLPrefix($baseLink);
            if (!Config::inst()->get('URLSegmentFilter', 'default_allow_multibyte')) {
                $helpText = _t('URLSegmentExtension.HELP',
                    ' Special characters are automatically converted or removed.');
                $urlSegmentField->setHelpText($helpText);
            }

            $fields->insertAfter($urlSegmentField, $this->getTarget());
        }
    }

    /**
     * Return the link for this {@link DataObject} object, with the {@link Director::baseURL()} included.
     *
     * @param string $action Optional controller action (method).
     *
     * @return string
     */
    public function Link($action = null)
    {
        return Controller::join_links(Director::baseURL(), $this->owner->RelativeLink($action));
    }

    /**
     * Get the absolute URL for this page, including protocol and host.
     *
     * @param string $action See {@link Link()}
     *
     * @return string
     */
    public function AbsoluteLink($action = null)
    {
        if ($this->owner->hasMethod('alternateAbsoluteLink')) {
            return $this->owner->alternateAbsoluteLink($action);
        } else {
            return Director::absoluteURL($this->owner->Link($action));
        }
    }

    /**
     * Return the link for this {@link DataObject} object relative to the SilverStripe root.
     *
     * @param string $action See {@link Link()}
     *
     * @return string
     */
    public function RelativeLink($action = null)
    {
        if ($this->owner->ParentID) {
            $base = $this->owner->Parent()->RelativeLink($this->owner->URLSegment);
        } elseif (!$action && $this->owner->URLSegment == RootURLController::get_homepage_link()) {
            // Unset base for root-level homepages.
            // Note: Homepages with action parameters (or $action === true)
            // need to retain their URLSegment.
            $base = null;
        } else {
            $base = $this->owner->URLSegment;
        }

        $this->owner->extend('updateRelativeLink', $base, $action);

        // Legacy support: If $action === true, retain URLSegment for homepages,
        // but don't append any action
        if ($action === true) {
            $action = null;
        }

        return Controller::join_links($base, '/', $action);
    }

    public function onBeforeWrite()
    {
        if(empty($this->owner->URLSegment)) {
            $this->owner->URLSegment = $this->generateURLSegment($this->owner->Title);
        }
    }

    /**
     * Generate a URL segment based on the title provided.
     *
     * @param string $title Page title.
     *
     * @return string Generated url segment
     */
    public function generateURLSegment($title)
    {
        $filter = URLSegmentFilter::create();
        $t = $filter->filter($title);

        // Fallback to generic page name if path is empty (= no valid, convertable characters)
        if (!$t || $t == '-' || $t == '-1') {
            $t = "page-$this->owner->ID";
        }

        // Hook for extensions
        $this->owner->extend('updateURLSegment', $t, $title);

        return $t;
    }

    /**
     * Returns TRUE if this object has a URLSegment value that does not conflict with any other objects.
     *
     * @return bool
     */
    public function validURLSegment()
    {
        if ($parent = $this->owner->Parent()) {
            if ($controller = ModelAsController::controller_for($parent)) {
                if ($controller instanceof Controller && $controller->hasAction($this->owner->URLSegment)) {
                    return false;
                }
            }
        }

        if (!$this->owner->ParentID) {
            if (class_exists($this->owner->URLSegment) && is_subclass_of($this->owner->URLSegment, 'RequestHandler')) {
                return false;
            }
        }

        $votes = array_filter(
            (array)$this->owner->extend('augmentValidURLSegment'),
            function ($v) {
                return !is_null($v);
            }
        );
        if ($votes) {
            return min($votes);
        }

        $segment = Convert::raw2sql($this->owner->URLSegment);
        $existingPage = (int)DataObject::get($this->owner->class)
            ->filter([
                'URLSegment' => $segment,
                'ID:not'     => $this->owner->ID
            ])->count();

        return ($existingPage === 0);
    }
}