<?php namespace StudioBonito\URLSegment\Forms;

use Controller;
use Convert;
use DataObject;
use Requirements;

/**
 * URLSegmentField.
 *
 * @author       Tom Densham <tom.densham@studiobonito.co.uk>
 * @copyright    Studio Bonito Ltd.
 */
class URLSegmentField extends \TextField
{
    /**
     * @var string
     */
    protected $helpText, $urlPrefix, $urlSuffix;

    /**
     * Name of the target class.
     *
     * @var string
     */
    protected $targetClass = 'SiteTree';

    /**
     * Template name to override FQCN generated template name.
     *
     * @var string
     */
    protected $template = 'URLSegmentField';

    /**
     * @var array
     */
    private static $allowed_actions = [
        'suggest'
    ];

    /**
     * Get the URL segment from the field.
     *
     * @return string
     */
    public function Value()
    {
        return rawurldecode($this->value);
    }

    /**
     * @return array
     */
    public function getAttributes()
    {
        return array_merge(
            parent::getAttributes(),
            [
                'data-prefix' => $this->getURLPrefix(),
                'data-suffix' => $this->getURLSuffix()
            ]
        );
    }

    /**
     * Overload the form field rendering to include custom javascript.
     *
     * @param array $properties
     *
     * @return string
     */
    public function Field($properties = [])
    {
        Requirements::javascript('silverstripe-urlsegment/assets/js/URLSegmentField.js');
        Requirements::add_i18n_javascript(CMS_DIR . '/javascript/lang', false, true);
        Requirements::css(CMS_DIR . "/css/screen.css");

        return parent::Field($properties);
    }

    /**
     * Use the target to generate a valid URL segment.
     *
     * @param $request
     *
     * @return string|void
     * @throws \SS_HTTPResponse_Exception
     */
    public function suggest($request)
    {
        if (!$request->getVar('value')) {
            return $this->httpError(405,
                _t('URLSegmentField.EMPTY', 'Please enter a URL Segment or click cancel')
            );
        }
        $target = $this->getTarget();

        // Same logic as SiteTree->onBeforeWrite
        $target->URLSegment = $target->generateURLSegment($request->getVar('value'));
        $count = 2;
        while (!$target->validURLSegment()) {
            $target->URLSegment = preg_replace('/-[0-9]+$/', null, $target->URLSegment) . '-' . $count;
            $count++;
        }

        Controller::curr()->getResponse()->addHeader('Content-Type', 'application/json');

        return Convert::raw2json(['value' => $target->URLSegment]);
    }

    /**
     * @param string $class
     */
    public function setTargetClass($class)
    {
        $this->targetClass = $class;
    }

    /**
     * The secondary text to show in the template
     *
     * @return string
     */
    public function getHelpText()
    {
        return $this->helpText;
    }

    /**
     * The secondary text to show
     *
     * @param string $string
     */
    public function setHelpText($string)
    {
        $this->helpText = $string;
    }

    /**
     * The url prefixes the url segment field to show in template
     *
     * @return string
     */
    public function getURLPrefix()
    {
        return $this->urlPrefix;
    }

    /**
     * The url that prefixes the url segment field
     *
     * @param string
     */
    public function setURLPrefix($url)
    {
        $this->urlPrefix = $url;
    }

    /**
     * The url suffixes the url segment field to show in template
     *
     * @return string
     */
    public function getURLSuffix()
    {
        return $this->urlSuffix;
    }

    /**
     * The url that suffixes the url segment field
     *
     * @param string $suffix
     */
    public function setURLSuffix($suffix)
    {
        $this->urlSuffix = $suffix;
    }

    /**
     * The field type
     *
     * @return string
     */
    public function Type()
    {
        return 'text urlsegment';
    }

    /**
     * The url for the url segment field
     *
     * @return string
     */
    public function getURL()
    {
        return Controller::join_links($this->getURLPrefix(), $this->Value(), $this->getURLSuffix());
    }

    /**
     * @return \DataObject
     */
    protected function getTarget()
    {
        $idField = $this->getForm()->Fields()->dataFieldByName('ID');

        return ($idField && $idField->Value()) ? DataObject::get_by_id($this->targetClass,
            $idField->Value()) : singleton($this->targetClass);
    }
}